package com.sensor.bridge;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * 기본 센서 파서 (범용)
 * 알 수 없는 센서 타입에 대한 기본 파싱 로직 제공
 */
public class DefaultParser implements SensorParser {
    
    private static final Logger logger = LoggerFactory.getLogger(DefaultParser.class);
    private static final String SENSOR_TYPE = "DEFAULT";
    
    @Override
    public double parseTemperature(SensorData data, ParamItem paramItem) {
        logger.debug("Default 파서 온도 파싱 시작: rawTem={}, floatValue={}, signedInt32Value={}", 
                    data.getRawTem(), data.getFloatValue(), data.getSignedInt32Value());
        
        // 1. param.dat 기반 보정 적용
        if (paramItem != null && paramItem.getScaleFactor() != 1.0) {
            double calibratedTemp = paramItem.calculateCalibratedTemperature(data.getRawTem());
            if (paramItem.isValidTemperature(calibratedTemp)) {
                logger.debug("Default param.dat 기반 보정 적용: {} -> {}°C", data.getRawTem(), calibratedTemp);
                return calibratedTemp;
            }
        }
        
        // 2. 기존 SensorDataParser 로직 사용 (호환성 유지)
        double temperature = SensorDataParser.parseTemperature(
            data.getRawTem(), 
            data.getFloatValue(), 
            data.getSignedInt32Value()
        );
        
        if (isValidTemperature(temperature)) {
            logger.debug("Default 기존 파싱 로직 사용: {}°C", temperature);
            return temperature;
        }
        
        // 3. 기본값 반환
        logger.warn("Default: 파싱 실패, 기본값 사용");
        return paramItem != null ? paramItem.getDefaultValue() : 25.0;
    }
    
    @Override
    public boolean isValid(SensorData data) {
        // 기본 센서 데이터 유효성 검증
        return data != null && 
               (data.getRawTem() >= 0 || 
                data.getFloatValue() >= -40.0 || 
                data.getSignedInt32Value() != 0);
    }
    
    @Override
    public String getSensorType() {
        return SENSOR_TYPE;
    }
    
    @Override
    public double getParsingQuality(SensorData data) {
        if (!isValid(data)) {
            return 0.0;
        }
        
        // 파싱 품질 점수 계산 (0.0 ~ 1.0)
        double quality = 0.0;
        
        // 기존 파싱 로직 성공 가능성
        quality += 0.6;
        
        // param.dat 보정 가능성
        quality += 0.2;
        
        // 기본값 사용 가능성
        quality += 0.2;
        
        return Math.min(quality, 1.0);
    }
    
    private boolean isValidTemperature(double temperature) {
        return temperature >= -40.0 && temperature <= 80.0;
    }
}
