# 센서 데이터 수집 시스템

리눅스 서버 기반 센서 데이터 수집 시스템을 Go 언어로 구현하여 기존 Java 기반 시스템을 대체하는 프로젝트입니다.

## 🏗️ 시스템 아키텍처

```
┌─────────────────┐    ┌─────────────────┐    ┌─────────────────┐
│   센서 디바이스   │────│   Java 브리지    │────│   Go 서버       │
│                 │    │   (RSNet SDK)   │    │   (REST API)    │
└─────────────────┘    └─────────────────┘    └─────────────────┘
                                                       │
                                                       ▼
┌─────────────────┐    ┌─────────────────┐    ┌─────────────────┐
│   웹 대시보드    │◄───│   PostgreSQL    │◄───│   Redis 캐시    │
│   (React)       │    │   (데이터 저장)   │    │   (실시간 캐시)  │
└─────────────────┘    └─────────────────┘    └─────────────────┘
```

## 🚀 빠른 시작

### Docker Compose로 전체 시스템 실행

```bash
# 저장소 클론
git clone <repository-url>
cd docker_sensor_server

# Docker Compose로 모든 서비스 실행
docker-compose up -d

# 서비스 상태 확인
docker-compose ps
```

### 개별 서비스 접근

- **웹 대시보드**: http://localhost:3000
- **Go API 서버**: http://localhost:8080
- **PostgreSQL**: localhost:5432
- **Redis**: localhost:6379
- **Java 브리지**: localhost:8020

## 📁 프로젝트 구조

```
docker_sensor_server/
├── sensor-server/          # Go 메인 서버
│   ├── cmd/sensor-server/
│   ├── internal/
│   │   ├── api/           # REST API 핸들러
│   │   ├── cache/         # Redis 캐시
│   │   ├── database/      # PostgreSQL 연동
│   │   ├── models/        # 데이터 모델
│   │   ├── server/        # 서버 설정
│   │   └── websocket/     # WebSocket 처리
│   ├── Dockerfile
│   └── go.mod
├── sensor-bridge/          # Java 센서 브리지
│   ├── src/main/java/
│   ├── pom.xml
│   └── Dockerfile
├── web-dashboard/          # React 웹 대시보드
│   ├── src/
│   ├── package.json
│   └── Dockerfile
├── docker-compose.yml      # 전체 시스템 설정
├── init-db.sql            # 데이터베이스 초기화
└── README.md
```

## 🔧 API 엔드포인트

### 센서 데이터
- `POST /api/sensor-data` - 센서 데이터 수신
- `GET /api/devices/:deviceId/latest` - 최신 데이터 조회
- `GET /api/devices/:deviceId/history` - 히스토리 데이터 조회

### 디바이스 관리
- `GET /api/devices` - 디바이스 목록 조회
- `GET /api/health` - 헬스체크

### WebSocket
- `GET /ws` - 실시간 데이터 스트림

## 🛠️ 개발 환경 설정

### Go 서버 개발

```bash
cd sensor-server

# 의존성 설치
go mod tidy

# 개발 서버 실행
go run cmd/sensor-server/main.go
```

### Java 브리지 개발

```bash
cd sensor-bridge

# Maven 빌드
mvn clean package

# 실행
java -jar target/sensor-bridge-1.0.0.jar
```

### React 대시보드 개발

```bash
cd web-dashboard

# 의존성 설치
npm install

# 개발 서버 실행
npm start
```

## 📊 데이터 모델

### SensorReading
```go
type SensorReading struct {
    ID           uint      `json:"id"`
    DeviceID     string    `json:"device_id"`
    NodeID       int       `json:"node_id"`
    Temperature  float64   `json:"temperature"`
    Humidity     float64   `json:"humidity"`
    Longitude    float64   `json:"longitude"`
    Latitude     float64   `json:"latitude"`
    RecordedTime time.Time `json:"recorded_time"`
    ReceivedTime time.Time `json:"received_time"`
    CreatedAt    time.Time `json:"created_at"`
}
```

### Device
```go
type Device struct {
    ID          uint      `json:"id"`
    DeviceID    string    `json:"device_id"`
    Name        string    `json:"name"`
    Description string    `json:"description"`
    Status      string    `json:"status"`
    LastSeen    time.Time `json:"last_seen"`
    CreatedAt   time.Time `json:"created_at"`
    UpdatedAt   time.Time `json:"updated_at"`
}
```

## 🔒 환경 변수

### Go 서버
```bash
DB_HOST=localhost
DB_PORT=5432
DB_USER=postgres
DB_PASSWORD=password
DB_NAME=sensor_db
REDIS_HOST=localhost
REDIS_PORT=6379
PORT=8080
GIN_MODE=debug
```

### Java 브리지
```bash
GO_SERVER_URL=http://localhost:8080
RS_SERVER_PORT=8020
```

### React 대시보드
```bash
REACT_APP_API_URL=http://localhost:8080
REACT_APP_WS_URL=ws://localhost:8080
```

## 📈 모니터링

### 로그 확인
```bash
# Go 서버 로그
docker-compose logs sensor-server

# Java 브리지 로그
docker-compose logs sensor-bridge

# 데이터베이스 로그
docker-compose logs postgres
```

### 성능 메트릭
- 응답 시간: < 100ms
- 처리량: > 5,000 TPS
- 가용성: > 99.9%
- 메모리 사용량: < 1GB

## 🐛 문제 해결

### 일반적인 문제

1. **포트 충돌**
   ```bash
   # 사용 중인 포트 확인
   netstat -tulpn | grep :8080
   
   # Docker Compose 재시작
   docker-compose down && docker-compose up -d
   ```

2. **데이터베이스 연결 실패**
   ```bash
   # PostgreSQL 컨테이너 상태 확인
   docker-compose ps postgres
   
   # 로그 확인
   docker-compose logs postgres
   ```

3. **Java SDK 문제**
   - `JavaSDKV2.2.2` 디렉토리가 올바른 위치에 있는지 확인
   - JAR 파일이 올바르게 복사되었는지 확인

## 📝 라이선스

이 프로젝트는 MIT 라이선스 하에 배포됩니다.

## 🤝 기여

1. Fork the Project
2. Create your Feature Branch (`git checkout -b feature/AmazingFeature`)
3. Commit your Changes (`git commit -m 'Add some AmazingFeature'`)
4. Push to the Branch (`git push origin feature/AmazingFeature`)
5. Open a Pull Request 