/**
 * 센서 값 포맷팅 함수
 * @param value 센서 값
 * @param precision 소수점 자릿수 (기본값: 1)
 * @returns 포맷팅된 값 또는 'N/A'
 */
export const formatSensorValue = (value: number | undefined, precision: number = 1): string => {
  if (value === undefined || value === null) {
    return 'N/A';
  }
  
  // 매우 작은 값(e-40 등) 처리
  if (Math.abs(value) < 1e-10) {
    return '0.0';
  }
  
  // 무한대 값 처리
  if (!isFinite(value)) {
    return 'N/A';
  }
  
  return value.toFixed(precision);
};

/**
 * 센서 값 검증 함수
 * @param value 센서 값
 * @param min 최소값
 * @param max 최대값
 * @returns 유효한 값인지 여부
 */
export const validateSensorValue = (value: number | undefined, min: number = -1000, max: number = 10000): boolean => {
  if (value === undefined || value === null) {
    return false;
  }
  
  if (!isFinite(value)) {
    return false;
  }
  
  if (Math.abs(value) < 1e-10) {
    return true; // 0 값은 유효
  }
  
  return value >= min && value <= max;
};

/**
 * 센서별 기본 정밀도 반환
 * @param sensorType 센서 타입
 * @returns 기본 정밀도
 */
export const getSensorPrecision = (sensorType: string): number => {
  const precisionMap: { [key: string]: number } = {
    temperature: 1,
    humidity: 1,
    pressure: 1,
    illumination: 0,
    pm10: 1,
    pm25: 1,
    tvoc: 1,
    co2: 0,
    o2: 1,
    co: 2,
  };
  
  return precisionMap[sensorType] || 1;
}; 