import React, { useState } from 'react';
import { useMonitoring } from '../hooks/useMonitoring';
import { logger } from '../utils/logger';

const SystemMonitor: React.FC = () => {
  const { metrics, performanceMetrics } = useMonitoring();
  const [showDetails, setShowDetails] = useState(false);
  const [logStats] = useState(() => logger.getLogStats());

  const getStatusColor = (status: string) => {
    switch (status) {
      case 'connected':
        return 'text-green-600';
      case 'connecting':
        return 'text-yellow-600';
      case 'disconnected':
        return 'text-red-600';
      default:
        return 'text-gray-600';
    }
  };

  const getQualityColor = (score: number) => {
    if (score >= 90) return 'text-green-600';
    if (score >= 70) return 'text-yellow-600';
    return 'text-red-600';
  };

  return (
    <div className="bg-white dark:bg-gray-800 rounded-lg shadow-sm border border-gray-200 dark:border-gray-700 p-6">
      <div className="flex items-center justify-between mb-4">
        <h3 className="text-lg font-medium text-gray-900 dark:text-white">시스템 모니터링</h3>
        <button
          onClick={() => setShowDetails(!showDetails)}
          className="text-sm text-blue-600 dark:text-blue-400 hover:underline"
        >
          {showDetails ? '간단히 보기' : '상세 보기'}
        </button>
      </div>

      <div className="grid grid-cols-2 md:grid-cols-4 gap-4 mb-4">
        {/* 메모리 사용량 */}
        <div className="text-center">
          <div className="text-2xl font-semibold text-gray-900 dark:text-white">
            {metrics.memoryUsage}%
          </div>
          <div className="text-sm text-gray-600 dark:text-gray-400">메모리</div>
        </div>

        {/* 네트워크 지연 */}
        <div className="text-center">
          <div className="text-2xl font-semibold text-gray-900 dark:text-white">
            {metrics.networkLatency}ms
          </div>
          <div className="text-sm text-gray-600 dark:text-gray-400">네트워크</div>
        </div>

        {/* WebSocket 상태 */}
        <div className="text-center">
          <div className={`text-2xl font-semibold ${getStatusColor(metrics.websocketStatus)}`}>
            {metrics.websocketStatus === 'connected' ? '연결됨' : 
             metrics.websocketStatus === 'connecting' ? '연결중' : '끊어짐'}
          </div>
          <div className="text-sm text-gray-600 dark:text-gray-400">WebSocket</div>
        </div>

        {/* 데이터 품질 */}
        <div className="text-center">
          <div className={`text-2xl font-semibold ${getQualityColor(metrics.dataQuality.qualityScore)}`}>
            {metrics.dataQuality.qualityScore}%
          </div>
          <div className="text-sm text-gray-600 dark:text-gray-400">데이터 품질</div>
        </div>
      </div>

      {showDetails && (
        <div className="space-y-4">
          {/* 센서 상태 */}
          <div className="border-t border-gray-200 dark:border-gray-700 pt-4">
            <h4 className="text-sm font-medium text-gray-900 dark:text-white mb-2">센서 상태</h4>
            <div className="grid grid-cols-3 gap-4 text-sm">
              <div>
                <span className="text-gray-600 dark:text-gray-400">전체:</span>
                <span className="ml-2 text-gray-900 dark:text-white">{metrics.sensorStatus.totalSensors}</span>
              </div>
              <div>
                <span className="text-gray-600 dark:text-gray-400">활성:</span>
                <span className="ml-2 text-green-600">{metrics.sensorStatus.activeSensors}</span>
              </div>
              <div>
                <span className="text-gray-600 dark:text-gray-400">비활성:</span>
                <span className="ml-2 text-red-600">{metrics.sensorStatus.inactiveSensors}</span>
              </div>
            </div>
          </div>

          {/* 성능 메트릭 */}
          <div className="border-t border-gray-200 dark:border-gray-700 pt-4">
            <h4 className="text-sm font-medium text-gray-900 dark:text-white mb-2">성능 메트릭</h4>
            <div className="grid grid-cols-3 gap-4 text-sm">
              <div>
                <span className="text-gray-600 dark:text-gray-400">API 응답:</span>
                <span className="ml-2 text-gray-900 dark:text-white">{performanceMetrics.apiCallTime}ms</span>
              </div>
              <div>
                <span className="text-gray-600 dark:text-gray-400">렌더링:</span>
                <span className="ml-2 text-gray-900 dark:text-white">{performanceMetrics.componentRenderTime}ms</span>
              </div>
              <div>
                <span className="text-gray-600 dark:text-gray-400">처리:</span>
                <span className="ml-2 text-gray-900 dark:text-white">{performanceMetrics.dataProcessingTime}ms</span>
              </div>
            </div>
          </div>

          {/* 로그 통계 */}
          <div className="border-t border-gray-200 dark:border-gray-700 pt-4">
            <h4 className="text-sm font-medium text-gray-900 dark:text-white mb-2">로그 통계</h4>
            <div className="grid grid-cols-2 md:grid-cols-5 gap-4 text-sm">
              <div>
                <span className="text-gray-600 dark:text-gray-400">전체:</span>
                <span className="ml-2 text-gray-900 dark:text-white">{logStats.total}</span>
              </div>
              <div>
                <span className="text-gray-600 dark:text-gray-400">Debug:</span>
                <span className="ml-2 text-blue-600">{logStats.byLevel.debug}</span>
              </div>
              <div>
                <span className="text-gray-600 dark:text-gray-400">Info:</span>
                <span className="ml-2 text-green-600">{logStats.byLevel.info}</span>
              </div>
              <div>
                <span className="text-gray-600 dark:text-gray-400">Warn:</span>
                <span className="ml-2 text-yellow-600">{logStats.byLevel.warn}</span>
              </div>
              <div>
                <span className="text-gray-600 dark:text-gray-400">Error:</span>
                <span className="ml-2 text-red-600">{logStats.byLevel.error}</span>
              </div>
            </div>
          </div>

          {/* 데이터 품질 상세 */}
          <div className="border-t border-gray-200 dark:border-gray-700 pt-4">
            <h4 className="text-sm font-medium text-gray-900 dark:text-white mb-2">데이터 품질 상세</h4>
            <div className="grid grid-cols-3 gap-4 text-sm">
              <div>
                <span className="text-gray-600 dark:text-gray-400">전체 레코드:</span>
                <span className="ml-2 text-gray-900 dark:text-white">{metrics.dataQuality.totalRecords}</span>
              </div>
              <div>
                <span className="text-gray-600 dark:text-gray-400">유효 레코드:</span>
                <span className="ml-2 text-green-600">{metrics.dataQuality.validRecords}</span>
              </div>
              <div>
                <span className="text-gray-600 dark:text-gray-400">무효 레코드:</span>
                <span className="ml-2 text-red-600">{metrics.dataQuality.invalidRecords}</span>
              </div>
            </div>
          </div>
        </div>
      )}
    </div>
  );
};

export default SystemMonitor; 