import { SensorData, Device, HistoryResponse } from '../services/api';

// 센서 데이터 타입 가드
export const isSensorData = (data: any): data is SensorData => {
  return data &&
         typeof data === 'object' &&
         typeof data.id === 'number' &&
         typeof data.device_id === 'string' &&
         typeof data.node_id === 'number' &&
         typeof data.temperature === 'number' &&
         typeof data.humidity === 'number' &&
         typeof data.longitude === 'number' &&
         typeof data.latitude === 'number' &&
         typeof data.recorded_time === 'string' &&
         typeof data.received_time === 'string';
};

// 디바이스 타입 가드
export const isDevice = (data: any): data is Device => {
  return data &&
         typeof data === 'object' &&
         typeof data.id === 'number' &&
         typeof data.device_id === 'string' &&
         typeof data.name === 'string' &&
         typeof data.description === 'string' && // description은 빈 문자열도 허용
         typeof data.status === 'string' &&
         typeof data.last_seen === 'string' &&
         typeof data.created_at === 'string' &&
         typeof data.updated_at === 'string';
};

// 히스토리 응답 타입 가드
export const isHistoryResponse = (data: any): data is HistoryResponse => {
  return data &&
         typeof data === 'object' &&
         typeof data.success === 'boolean' &&
         Array.isArray(data.data) &&
         typeof data.total === 'number';
};

// 센서 데이터 배열 타입 가드
export const isSensorDataArray = (data: any): data is SensorData[] => {
  return Array.isArray(data) && data.every(item => isSensorData(item));
};

// 디바이스 배열 타입 가드
export const isDeviceArray = (data: any): data is Device[] => {
  return Array.isArray(data) && data.every(item => isDevice(item));
};

// 숫자 타입 가드
export const isNumber = (value: any): value is number => {
  return typeof value === 'number' && !isNaN(value) && isFinite(value);
};

// 문자열 타입 가드
export const isString = (value: any): value is string => {
  return typeof value === 'string' && value.length > 0;
};

// 객체 타입 가드
export const isObject = (value: any): value is Record<string, any> => {
  return value !== null && typeof value === 'object' && !Array.isArray(value);
};

// 배열 타입 가드
export const isArray = (value: any): value is any[] => {
  return Array.isArray(value);
};

// API 응답 타입 가드
export const isApiResponse = (data: any): boolean => {
  return data && typeof data === 'object';
};

// 차트 데이터 타입 가드
export const isChartData = (data: any): boolean => {
  return Array.isArray(data) && 
         data.length > 0 && 
         data.every(item => 
           item && 
           typeof item === 'object' && 
           typeof item.device_id === 'string'
         );
};

// 안전한 숫자 변환 함수
export const toSafeNumber = (value: any, defaultValue: number = 0): number => {
  if (value === null || value === undefined) {
    return defaultValue;
  }
  
  const num = Number(value);
  if (isNumber(num)) {
    return num;
  }
  
  return defaultValue;
};

// 안전한 문자열 변환 함수
export const toSafeString = (value: any, defaultValue: string = ''): string => {
  if (value === null || value === undefined) {
    return defaultValue;
  }
  
  const str = String(value);
  if (isString(str)) {
    return str;
  }
  
  return defaultValue;
};

// 객체에서 안전하게 값 추출하는 함수
export const safeGet = <T>(obj: any, key: string, defaultValue: T): T => {
  if (!isObject(obj) || !(key in obj)) {
    return defaultValue;
  }
  
  return obj[key] ?? defaultValue;
};

// 배열에서 안전하게 인덱스 접근하는 함수
export const safeGetAt = <T>(arr: any[], index: number, defaultValue: T): T => {
  if (!isArray(arr) || index < 0 || index >= arr.length) {
    return defaultValue;
  }
  
  return arr[index] ?? defaultValue;
}; 