import React from 'react';
import { render, screen } from '@testing-library/react';
import SensorCard from '../SensorCard';

describe('SensorCard', () => {
  const defaultProps = {
    title: '테스트 센서',
    value: 25.5,
    unit: '°C',
    icon: '🌡️',
    bgColor: 'bg-blue-100',
    precision: 1
  };

  it('should render sensor card with all props', () => {
    render(<SensorCard {...defaultProps} />);
    
    expect(screen.getByText('테스트 센서')).toBeInTheDocument();
    expect(screen.getByText('25.5°C')).toBeInTheDocument();
    expect(screen.getByText('🌡️')).toBeInTheDocument();
  });

  it('should format value with correct precision', () => {
    render(<SensorCard {...defaultProps} value={25.123} precision={2} />);
    
    expect(screen.getByText('25.12°C')).toBeInTheDocument();
  });

  it('should handle zero values', () => {
    render(<SensorCard {...defaultProps} value={0} />);
    
    expect(screen.getByText('0.0°C')).toBeInTheDocument();
  });

  it('should handle very small values', () => {
    render(<SensorCard {...defaultProps} value={1e-40} />);
    
    expect(screen.getByText('0.0°C')).toBeInTheDocument();
  });

  it('should handle undefined values', () => {
    render(<SensorCard {...defaultProps} value={undefined} />);
    
    expect(screen.getByText('N/A°C')).toBeInTheDocument();
  });

  it('should handle null values', () => {
    render(<SensorCard {...defaultProps} value={null as any} />);
    
    expect(screen.getByText('N/A°C')).toBeInTheDocument();
  });

  it('should handle infinite values', () => {
    render(<SensorCard {...defaultProps} value={Infinity} />);
    
    expect(screen.getByText('N/A°C')).toBeInTheDocument();
  });

  it('should handle NaN values', () => {
    render(<SensorCard {...defaultProps} value={NaN} />);
    
    expect(screen.getByText('N/A°C')).toBeInTheDocument();
  });

  it('should apply custom background color', () => {
    render(<SensorCard {...defaultProps} bgColor="bg-red-100" />);
    
    const card = screen.getByText('테스트 센서').closest('div');
    expect(card).toHaveClass('bg-red-100');
  });

  it('should render without unit', () => {
    render(<SensorCard {...defaultProps} unit="" />);
    
    expect(screen.getByText('25.5')).toBeInTheDocument();
  });

  it('should render without icon', () => {
    render(<SensorCard {...defaultProps} icon="" />);
    
    expect(screen.queryByText('🌡️')).not.toBeInTheDocument();
  });

  it('should handle different sensor types', () => {
    const testCases = [
      { title: '온도', value: 25.5, unit: '°C', icon: '🌡️' },
      { title: '습도', value: 60.0, unit: '%', icon: '💧' },
      { title: 'PM10', value: 15.2, unit: 'μg/m³', icon: '🌫️' },
      { title: '기압', value: 1013.25, unit: 'hPa', icon: '🌪️' }
    ];

    testCases.forEach(({ title, value, unit, icon }) => {
      const { unmount } = render(
        <SensorCard 
          title={title}
          value={value}
          unit={unit}
          icon={icon}
          bgColor="bg-gray-100"
          precision={1}
        />
      );

      expect(screen.getByText(title)).toBeInTheDocument();
      expect(screen.getByText(`${value}${unit}`)).toBeInTheDocument();
      expect(screen.getByText(icon)).toBeInTheDocument();

      unmount();
    });
  });

  it('should handle precision 0', () => {
    render(<SensorCard {...defaultProps} value={25.7} precision={0} />);
    
    expect(screen.getByText('26°C')).toBeInTheDocument();
  });

  it('should handle negative values', () => {
    render(<SensorCard {...defaultProps} value={-10.5} />);
    
    expect(screen.getByText('-10.5°C')).toBeInTheDocument();
  });
}); 