import React from 'react';
import { ConnectionStatus } from '../hooks/useRealTimeData';

interface StatusIndicatorProps {
  isConnected: boolean;
  connectionStatus?: ConnectionStatus;
  dataSource: 'websocket' | 'polling';
  lastUpdate?: string;
  className?: string;
  error?: string | null;
  retryCount?: Map<string, number>;
}

const StatusIndicator: React.FC<StatusIndicatorProps> = ({
  isConnected,
  connectionStatus,
  dataSource,
  lastUpdate,
  className = '',
  error,
  retryCount
}) => {
  const getStatusColor = () => {
    if (connectionStatus) {
      switch (connectionStatus) {
        case 'connected':
          return 'text-green-600';
        case 'connecting':
          return 'text-yellow-600';
        case 'retrying':
          return 'text-orange-600';
        case 'error':
        case 'disconnected':
          return 'text-red-600';
        default:
          return 'text-gray-600';
      }
    }
    
    if (!isConnected) return 'text-red-600';
    return dataSource === 'websocket' ? 'text-green-600' : 'text-yellow-600';
  };

  const getStatusText = () => {
    if (connectionStatus) {
      switch (connectionStatus) {
        case 'connected':
          return '연결됨';
        case 'connecting':
          return '연결 중...';
        case 'retrying':
          return '재시도 중...';
        case 'error':
          return '연결 오류';
        case 'disconnected':
          return '연결 끊김';
        default:
          return '알 수 없음';
      }
    }
    
    if (!isConnected) return '연결 끊김';
    return dataSource === 'websocket' ? '실시간 연결' : '폴링 모드';
  };

  const getStatusIcon = () => {
    if (connectionStatus) {
      switch (connectionStatus) {
        case 'connected':
          return (
            <svg className="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
              <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M5 13l4 4L19 7" />
            </svg>
          );
        case 'connecting':
        case 'retrying':
          return (
            <svg className="w-4 h-4 animate-spin" fill="none" stroke="currentColor" viewBox="0 0 24 24">
              <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M4 4v5h.582m15.356 2A8.001 8.001 0 004.582 9m0 0H9m11 11v-5h-.581m0 0a8.003 8.003 0 01-15.357-2m15.357 2H15" />
            </svg>
          );
        case 'error':
        case 'disconnected':
          return (
            <svg className="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
              <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M12 8v4m0 4h.01M21 12a9 9 0 11-18 0 9 9 0 0118 0z" />
            </svg>
          );
        default:
          return (
            <svg className="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
              <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M8.228 9c.549-1.165 2.03-2 3.772-2 2.21 0 4 1.343 4 3 0 1.4-1.278 2.575-3.006 2.907-.542.104-.994.54-.994 1.093m0 3h.01M21 12a9 9 0 11-18 0 9 9 0 0118 0z" />
            </svg>
          );
      }
    }
    
    if (dataSource === 'websocket') {
      return (
        <svg className="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
          <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M13 10V3L4 14h7v7l9-11h-7z" />
        </svg>
      );
    }
    return (
      <svg className="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
        <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M12 8v4l3 3m6-3a9 9 0 11-18 0 9 9 0 0118 0z" />
      </svg>
    );
  };

  const getDataSourceText = () => {
    return dataSource === 'websocket' ? 'WebSocket' : '폴링';
  };

  const getRetryInfo = () => {
    if (!retryCount || retryCount.size === 0) return null;
    
    const totalRetries = Array.from(retryCount.values()).reduce((sum, count) => sum + count, 0);
    if (totalRetries === 0) return null;
    
    return (
      <div className="text-xs text-orange-600 bg-orange-50 px-2 py-1 rounded">
        재시도: {totalRetries}회
      </div>
    );
  };

  return (
    <div className={`flex items-center space-x-3 text-sm ${className}`}>
      {/* 연결 상태 */}
      <div className={`flex items-center space-x-1 ${getStatusColor()}`}>
        <div className={`w-2 h-2 rounded-full ${
          connectionStatus === 'connected' ? 'bg-green-500' : 
          connectionStatus === 'connecting' || connectionStatus === 'retrying' ? 'bg-yellow-500' :
          connectionStatus === 'error' || connectionStatus === 'disconnected' ? 'bg-red-500' :
          isConnected ? 'bg-green-500' : 'bg-red-500'
        }`} />
        {getStatusIcon()}
        <span className="font-medium">{getStatusText()}</span>
      </div>

      {/* 데이터 소스 */}
      <div className="flex items-center space-x-1 text-gray-500">
        <span>{getDataSourceText()}</span>
      </div>

      {/* 재시도 정보 */}
      {getRetryInfo()}

      {/* 마지막 업데이트 */}
      {lastUpdate && (
        <div className="text-gray-400 text-xs">
          마지막 업데이트: {new Date(lastUpdate).toLocaleTimeString()}
        </div>
      )}

      {/* 에러 메시지 */}
      {error && (
        <div className="text-red-600 text-xs bg-red-50 px-2 py-1 rounded max-w-xs truncate" title={error}>
          {error}
        </div>
      )}
    </div>
  );
};

export default StatusIndicator; 