package com.sensor.bridge;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import java.util.HashMap;
import java.util.Map;

/**
 * 센서 파서 팩토리
 * 센서 타입에 따라 적절한 파서를 생성하고 관리
 */
public class SensorParserFactory {
    
    private static final Logger logger = LoggerFactory.getLogger(SensorParserFactory.class);
    
    // 파서 인스턴스 캐시
    private static final Map<String, SensorParser> parserCache = new HashMap<>();
    
    // 기본 파서 인스턴스
    private static final DefaultParser defaultParser = new DefaultParser();
    
    static {
        // 기본 파서들을 캐시에 등록
        registerParser("SHT30", new SHT30Parser());
        registerParser("BME680", new BME680Parser());
        registerParser("DEFAULT", defaultParser);
    }
    
    /**
     * 센서 타입에 따른 파서 생성
     * @param sensorType 센서 타입
     * @return 적절한 센서 파서
     */
    public static SensorParser createParser(String sensorType) {
        if (sensorType == null || sensorType.trim().isEmpty()) {
            logger.warn("센서 타입이 null이거나 비어있음, 기본 파서 사용");
            return defaultParser;
        }
        
        String normalizedType = sensorType.trim().toUpperCase();
        
        // 캐시된 파서 반환
        SensorParser cachedParser = parserCache.get(normalizedType);
        if (cachedParser != null) {
            logger.debug("캐시된 파서 사용: {}", normalizedType);
            return cachedParser;
        }
        
        // 새로운 파서 생성
        SensorParser newParser = createNewParser(normalizedType);
        if (newParser != null) {
            registerParser(normalizedType, newParser);
            return newParser;
        }
        
        // 기본 파서 반환
        logger.debug("알 수 없는 센서 타입: {}, 기본 파서 사용", normalizedType);
        return defaultParser;
    }
    
    /**
     * 센서 데이터를 기반으로 최적의 파서 선택
     * @param data 센서 데이터
     * @return 최적의 센서 파서
     */
    public static SensorParser selectBestParser(SensorData data) {
        if (data == null) {
            return defaultParser;
        }
        
        // 각 파서의 품질 점수 계산
        Map<String, Double> qualityScores = new HashMap<>();
        
        for (SensorParser parser : parserCache.values()) {
            double quality = parser.getParsingQuality(data);
            qualityScores.put(parser.getSensorType(), quality);
            logger.debug("파서 {} 품질 점수: {}", parser.getSensorType(), quality);
        }
        
        // 최고 품질 점수를 가진 파서 선택
        String bestParserType = qualityScores.entrySet().stream()
            .max(Map.Entry.comparingByValue())
            .map(Map.Entry::getKey)
            .orElse("DEFAULT");
        
        logger.debug("최적 파서 선택: {} (품질 점수: {})", bestParserType, qualityScores.get(bestParserType));
        return parserCache.get(bestParserType);
    }
    
    /**
     * 새로운 파서 등록
     * @param sensorType 센서 타입
     * @param parser 센서 파서
     */
    public static void registerParser(String sensorType, SensorParser parser) {
        if (sensorType != null && parser != null) {
            String normalizedType = sensorType.trim().toUpperCase();
            parserCache.put(normalizedType, parser);
            logger.info("새로운 파서 등록: {} -> {}", normalizedType, parser.getClass().getSimpleName());
        }
    }
    
    /**
     * 등록된 파서 목록 반환
     * @return 등록된 파서 타입 목록
     */
    public static String[] getRegisteredParserTypes() {
        return parserCache.keySet().toArray(new String[0]);
    }
    
    /**
     * 특정 센서 타입의 파서 존재 여부 확인
     * @param sensorType 센서 타입
     * @return 파서 존재 여부
     */
    public static boolean hasParser(String sensorType) {
        if (sensorType == null) {
            return false;
        }
        return parserCache.containsKey(sensorType.trim().toUpperCase());
    }
    
    private static SensorParser createNewParser(String sensorType) {
        // 센서 타입에 따른 새로운 파서 생성 로직
        switch (sensorType) {
            case "SHT30":
                return new SHT30Parser();
            case "BME680":
                return new BME680Parser();
            case "DHT22":
            case "DHT11":
                // DHT 시리즈 센서용 파서 (향후 구현)
                logger.info("DHT 시리즈 센서 파서는 아직 구현되지 않음, 기본 파서 사용");
                return defaultParser;
            case "DS18B20":
                // DS18B20 온도 센서용 파서 (향후 구현)
                logger.info("DS18B20 센서 파서는 아직 구현되지 않음, 기본 파서 사용");
                return defaultParser;
            default:
                logger.debug("지원되지 않는 센서 타입: {}", sensorType);
                return null;
        }
    }
}
