#!/bin/bash

# PostgreSQL 백업 스크립트
# 사용법: ./backup.sh [full|incremental]

set -e

# 색상 정의
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
NC='\033[0m'

# 로그 함수
log_info() {
    echo -e "${GREEN}[INFO]${NC} $1"
}

log_warn() {
    echo -e "${YELLOW}[WARN]${NC} $1"
}

log_error() {
    echo -e "${RED}[ERROR]${NC} $1"
}

# 설정
BACKUP_TYPE=${1:-full}
BACKUP_DIR="./backups"
TIMESTAMP=$(date +"%Y%m%d_%H%M%S")
DB_NAME="sensor_db"
DB_USER="postgres"
DB_HOST="localhost"
DB_PORT="5432"

# 백업 디렉토리 생성
mkdir -p "$BACKUP_DIR"

# 백업 파일명 생성
case $BACKUP_TYPE in
    "full")
        BACKUP_FILE="$BACKUP_DIR/full_backup_$TIMESTAMP.sql"
        log_info "전체 백업을 시작합니다: $BACKUP_FILE"
        ;;
    "incremental")
        BACKUP_FILE="$BACKUP_DIR/incremental_backup_$TIMESTAMP.sql"
        log_info "증분 백업을 시작합니다: $BACKUP_FILE"
        ;;
    *)
        log_error "잘못된 백업 타입입니다. 사용법: $0 [full|incremental]"
        exit 1
        ;;
esac

# Docker 컨테이너 확인
if ! docker-compose ps postgres | grep -q "Up"; then
    log_error "PostgreSQL 컨테이너가 실행 중이지 않습니다."
    exit 1
fi

# 백업 실행
log_info "PostgreSQL 백업을 시작합니다..."

if [ "$BACKUP_TYPE" = "full" ]; then
    # 전체 백업
    docker-compose exec -T postgres pg_dump \
        -U "$DB_USER" \
        -h "$DB_HOST" \
        -p "$DB_PORT" \
        -d "$DB_NAME" \
        --clean \
        --if-exists \
        --create \
        --verbose \
        > "$BACKUP_FILE"
else
    # 증분 백업 (최근 24시간 데이터만)
    docker-compose exec -T postgres pg_dump \
        -U "$DB_USER" \
        -h "$DB_HOST" \
        -p "$DB_PORT" \
        -d "$DB_NAME" \
        --data-only \
        --table="sensor_readings" \
        --where="created_at >= NOW() - INTERVAL '24 hours'" \
        --verbose \
        > "$BACKUP_FILE"
fi

# 백업 파일 압축
log_info "백업 파일을 압축합니다..."
gzip "$BACKUP_FILE"
BACKUP_FILE_GZ="$BACKUP_FILE.gz"

# 백업 파일 크기 확인
BACKUP_SIZE=$(du -h "$BACKUP_FILE_GZ" | cut -f1)
log_info "백업 완료: $BACKUP_FILE_GZ (크기: $BACKUP_SIZE)"

# 오래된 백업 파일 정리 (30일 이상)
log_info "오래된 백업 파일을 정리합니다..."
find "$BACKUP_DIR" -name "*.sql.gz" -mtime +30 -delete

# 백업 파일 목록 출력
log_info "현재 백업 파일 목록:"
ls -lh "$BACKUP_DIR"/*.sql.gz 2>/dev/null || log_warn "백업 파일이 없습니다."

# 백업 완료
log_info "백업이 성공적으로 완료되었습니다!" 