# Java Spring(전자정부프레임워크) 센서 시스템 개발 가이드 - Part 3

## 🚀 배포 및 실행

### 1. 개발 환경 실행
```bash
# 프로젝트 디렉토리로 이동
cd sensor-spring-app

# Maven 의존성 설치
mvn clean install

# Spring Boot 애플리케이션 실행
mvn spring-boot:run
```

### 2. Docker 빌드 및 실행
```bash
# 전체 시스템 실행
docker-compose up -d

# 특정 서비스만 실행
docker-compose up -d sensor-spring-app

# 로그 확인
docker-compose logs -f sensor-spring-app
```

### 3. 운영 환경 배포
```bash
# JAR 파일 빌드
mvn clean package -DskipTests

# JAR 파일 실행
java -jar target/sensor-spring-app-1.0.0.jar \
  --spring.profiles.active=prod \
  --server.port=8080
```

---

## 📊 모니터링 및 로깅

### 1. Spring Actuator 엔드포인트
- **Health Check**: `/actuator/health`
- **Info**: `/actuator/info`
- **Metrics**: `/actuator/metrics`
- **Prometheus**: `/actuator/prometheus`

### 2. 로깅 설정
```yaml
logging:
  pattern:
    console: "%d{yyyy-MM-dd HH:mm:ss} [%thread] %-5level %logger{36} - %msg%n"
    file: "%d{yyyy-MM-dd HH:mm:ss} [%thread] %-5level %logger{36} - %msg%n"
  file:
    name: logs/sensor-spring-app.log
  level:
    com.sensor.spring: INFO
    org.springframework.web: INFO
    org.hibernate.SQL: DEBUG
```

---

## 🧪 테스트

### 1. 단위 테스트
```java
@SpringBootTest
class SensorDataServiceTest {
    
    @Autowired
    private SensorDataService sensorDataService;
    
    @Test
    void testSaveSensorData() {
        SensorDataRequest request = SensorDataRequest.builder()
            .deviceId("TEST_DEVICE")
            .temperature(25.5)
            .humidity(60.0)
            .build();
        
        SensorDataResponse response = sensorDataService.saveSensorData(request);
        
        assertNotNull(response);
        assertEquals("TEST_DEVICE", response.getDeviceId());
        assertEquals(25.5, response.getTemperature());
    }
}
```

### 2. 통합 테스트
```java
@SpringBootTest(webEnvironment = SpringBootTest.WebEnvironment.RANDOM_PORT)
@AutoConfigureTestDatabase(replace = AutoConfigureTestDatabase.Replace.NONE)
class SensorDataControllerIntegrationTest {
    
    @Autowired
    private TestRestTemplate restTemplate;
    
    @Test
    void testSaveSensorDataEndpoint() {
        SensorDataRequest request = new SensorDataRequest();
        request.setDeviceId("TEST_DEVICE");
        request.setTemperature(25.5);
        
        ResponseEntity<SensorDataResponse> response = restTemplate.postForEntity(
            "/api/sensor-data", request, SensorDataResponse.class);
        
        assertEquals(HttpStatus.CREATED, response.getStatusCode());
        assertNotNull(response.getBody());
    }
}
```

---

## 🔒 보안 설정

### 1. Spring Security 설정
```java
@Configuration
@EnableWebSecurity
public class SecurityConfig extends WebSecurityConfigurerAdapter {
    
    @Override
    protected void configure(HttpSecurity http) throws Exception {
        http
            .authorizeRequests()
                .antMatchers("/api/public/**").permitAll()
                .antMatchers("/api/**").authenticated()
                .anyRequest().permitAll()
            .and()
            .csrf().disable()
            .httpBasic();
    }
}
```

### 2. CORS 설정
```java
@Configuration
public class CorsConfig {
    
    @Bean
    public CorsConfigurationSource corsConfigurationSource() {
        CorsConfiguration configuration = new CorsConfiguration();
        configuration.setAllowedOrigins(Arrays.asList("*"));
        configuration.setAllowedMethods(Arrays.asList("GET", "POST", "PUT", "DELETE"));
        configuration.setAllowedHeaders(Arrays.asList("*"));
        
        UrlBasedCorsConfigurationSource source = new UrlBasedCorsConfigurationSource();
        source.registerCorsConfiguration("/**", configuration);
        return source;
    }
}
```

---

## 📈 성능 최적화

### 1. 캐싱 전략
- **Redis 캐싱**: 최신 데이터, 디바이스 목록
- **JPA 2nd Level Cache**: 엔티티 캐싱
- **HTTP 응답 캐싱**: 정적 리소스

### 2. 데이터베이스 최적화
- **인덱스**: device_id, recorded_time
- **파티셔닝**: 날짜별 테이블 파티셔닝
- **Connection Pool**: HikariCP 설정

### 3. 비동기 처리
- **@Async**: 센서 데이터 저장
- **@Scheduled**: 정기적인 데이터 정리
- **WebSocket**: 실시간 데이터 스트리밍

---

## 🐛 문제 해결

### 1. 일반적인 문제

#### **포트 충돌**
```bash
# 사용 중인 포트 확인
netstat -tulpn | grep :8080

# Spring Boot 애플리케이션 중지
pkill -f "sensor-spring-app"
```

#### **데이터베이스 연결 실패**
```bash
# PostgreSQL 상태 확인
docker-compose ps postgres

# 연결 테스트
docker exec -it sensor-postgres psql -U postgres -d sensor_db
```

#### **메모리 부족**
```bash
# JVM 메모리 설정
java -Xmx2g -Xms1g -jar target/sensor-spring-app-1.0.0.jar
```

### 2. 로그 분석
```bash
# 애플리케이션 로그 확인
tail -f logs/sensor-spring-app.log

# 에러 로그 필터링
grep "ERROR" logs/sensor-spring-app.log

# 특정 디바이스 로그
grep "deviceId=15328737" logs/sensor-spring-app.log
```

---

## 📝 향후 개선 계획

### 1. 단기 개선 (1-2주)
- [ ] Spring Security 인증/인가 시스템 구축
- [ ] API 문서화 (Swagger/OpenAPI)
- [ ] 단위 테스트 커버리지 향상

### 2. 중기 개선 (1-2개월)
- [ ] 실시간 알림 시스템 (WebSocket)
- [ ] 데이터 검증 및 이상치 탐지
- [ ] 성능 모니터링 대시보드

### 3. 장기 개선 (3-6개월)
- [ ] 머신러닝 기반 예측 시스템
- [ ] 마이크로서비스 아키텍처 전환
- [ ] 클라우드 네이티브 배포

---

## 📚 참고 자료

### 1. 공식 문서
- [Spring Boot Reference](https://docs.spring.io/spring-boot/docs/current/reference/html/)
- [전자정부프레임워크 가이드](https://www.egovframe.go.kr/home/sub.do?menuNo=9)
- [Spring Data JPA](https://docs.spring.io/spring-data/jpa/docs/current/reference/html/)

### 2. 관련 라이브러리
- [Thymeleaf](https://www.thymeleaf.org/documentation.html)
- [Chart.js](https://www.chartjs.org/docs/latest/)
- [Bootstrap](https://getbootstrap.com/docs/)

---

## ✅ 체크리스트

### **개발 환경 설정**
- [x] Java 11 설치
- [x] Maven 3.6.x 설치
- [x] IDE 설정
- [x] 프로젝트 구조 생성

### **핵심 기능 구현**
- [x] JPA 엔티티 정의
- [x] Repository 계층 구현
- [x] Service 계층 구현
- [x] Controller 계층 구현
- [x] 웹 인터페이스 구현

### **설정 및 배포**
- [x] application.yml 설정
- [x] Docker 설정
- [x] 데이터베이스 연동
- [x] Redis 캐싱 설정

### **테스트 및 모니터링**
- [x] 단위 테스트 작성
- [x] 통합 테스트 작성
- [x] 로깅 설정
- [x] Actuator 설정

---

## 🔄 기존 시스템과의 연동 시나리오

### 1. 점진적 마이그레이션
```
1단계: Spring Boot 애플리케이션 추가
├── 기존 Go 서버 유지 (포트: 8080)
├── 새로운 Spring Boot 추가 (포트: 8081)
└── 로드 밸런서로 트래픽 분산

2단계: 기능별 전환
├── 센서 데이터 수집: Spring Boot로 전환
├── 데이터 조회: Spring Boot로 전환
└── 웹 대시보드: Spring Boot로 전환

3단계: 완전 전환
├── Go 서버 중단
├── Spring Boot를 메인 서비스로 설정
└── 포트를 8080으로 변경
```

### 2. 데이터 마이그레이션
```sql
-- 기존 테이블 구조 유지
-- 새로운 인덱스 추가
CREATE INDEX idx_sensor_readings_device_time 
ON sensor_readings(device_id, recorded_time);

-- 파티셔닝 테이블 생성 (선택사항)
CREATE TABLE sensor_readings_2024 PARTITION OF sensor_readings
FOR VALUES FROM ('2024-01-01') TO ('2025-01-01');
```

### 3. API 호환성 유지
```java
// 기존 Go 서버와 동일한 API 엔드포인트 제공
@RestController
@RequestMapping("/api")
public class LegacyApiController {
    
    @PostMapping("/sensor-data")
    public ResponseEntity<SensorDataResponse> legacySensorData(
            @RequestBody LegacySensorDataRequest request) {
        // 기존 형식의 요청을 새로운 형식으로 변환
        SensorDataRequest newRequest = convertLegacyRequest(request);
        return sensorDataService.saveSensorData(newRequest);
    }
}
```

---

## 🎯 개발 우선순위

### **Phase 1: 기본 구조 (1-2주)**
1. Spring Boot 프로젝트 생성
2. 기본 엔티티 및 Repository 구현
3. 간단한 REST API 구현
4. 데이터베이스 연결 테스트

### **Phase 2: 핵심 기능 (2-3주)**
1. 센서 데이터 수집 API 완성
2. 데이터 조회 및 히스토리 API 구현
3. Redis 캐싱 시스템 구축
4. 기본 웹 대시보드 구현

### **Phase 3: 고도화 (3-4주)**
1. 전자정부프레임워크 통합
2. 보안 시스템 구축
3. 모니터링 및 로깅 시스템
4. 성능 최적화

### **Phase 4: 테스트 및 배포 (1-2주)**
1. 단위/통합 테스트 작성
2. Docker 컨테이너화
3. 운영 환경 배포
4. 기존 시스템과의 연동 테스트

---

## 💡 개발 팁

### 1. Spring Boot 개발 팁
- **프로파일 활용**: dev, test, prod 환경별 설정 분리
- **자동 설정**: Spring Boot의 자동 설정 기능 최대한 활용
- **의존성 관리**: Maven BOM을 통한 버전 관리

### 2. JPA 사용 팁
- **N+1 문제 방지**: fetch join, @EntityGraph 활용
- **벌크 연산**: JpaRepository.saveAll() 활용
- **트랜잭션 관리**: @Transactional 범위 최소화

### 3. 성능 최적화 팁
- **캐싱 전략**: Redis + JPA 2nd Level Cache 조합
- **비동기 처리**: @Async + CompletableFuture 활용
- **데이터베이스 최적화**: 인덱스, 파티셔닝, 커넥션 풀

---

## 🚨 주의사항

### 1. 보안 관련
- **입력 검증**: 모든 사용자 입력에 대한 검증 필수
- **SQL 인젝션**: JPA 사용으로 자동 방지
- **XSS 방지**: Thymeleaf의 기본 이스케이프 활용

### 2. 성능 관련
- **메모리 사용량**: JVM 힙 메모리 모니터링 필수
- **데이터베이스 연결**: 커넥션 풀 크기 적절히 설정
- **캐시 무효화**: 데이터 변경 시 캐시 무효화 전략 수립

### 3. 운영 관련
- **로그 관리**: 로그 파일 크기 및 보관 기간 설정
- **모니터링**: Spring Actuator + Prometheus 연동
- **백업**: 데이터베이스 정기 백업 및 복구 테스트

---

## 📞 지원 및 문의

### **개발 지원**
- **기술 문서**: 프로젝트 내 README.md 참조
- **API 문서**: Swagger UI (/swagger-ui.html)
- **모니터링**: Spring Actuator 엔드포인트 활용

### **문제 해결**
- **로그 분석**: 애플리케이션 로그 및 시스템 로그 확인
- **성능 분석**: JVM 메트릭 및 데이터베이스 쿼리 분석
- **네트워크 분석**: 포트 및 연결 상태 확인

---

**작성일**: 2024년 12월  
**작성자**: AI Assistant  
**버전**: 1.0.0  

---

*이 가이드는 Java Spring(전자정부프레임워크)으로 센서 정보를 수집하고 관리하는 웹프로그램을 개발하기 위한 것입니다.  
추가 질문이나 명확화가 필요한 부분이 있으면 언제든 연락주세요.*
