import { formatSensorValue, validateSensorValue, getSensorPrecision } from '../formatters';

describe('formatters', () => {
  describe('formatSensorValue', () => {
    it('should format valid numbers with default precision', () => {
      expect(formatSensorValue(25.123456)).toBe('25.1');
      expect(formatSensorValue(0)).toBe('0.0');
      expect(formatSensorValue(-10.5)).toBe('-10.5');
    });

    it('should format numbers with custom precision', () => {
      expect(formatSensorValue(25.123456, 2)).toBe('25.12');
      expect(formatSensorValue(25.123456, 0)).toBe('25');
      expect(formatSensorValue(25.123456, 4)).toBe('25.1235');
    });

    it('should handle very small values (e-40)', () => {
      expect(formatSensorValue(1e-40)).toBe('0.0');
      expect(formatSensorValue(1e-15)).toBe('0.0');
      expect(formatSensorValue(1e-10)).toBe('0.0');
    });

    it('should handle undefined and null values', () => {
      expect(formatSensorValue(undefined)).toBe('N/A');
      expect(formatSensorValue(null as any)).toBe('N/A');
    });

    it('should handle infinite values', () => {
      expect(formatSensorValue(Infinity)).toBe('N/A');
      expect(formatSensorValue(-Infinity)).toBe('N/A');
    });

    it('should handle NaN values', () => {
      expect(formatSensorValue(NaN)).toBe('N/A');
    });
  });

  describe('validateSensorValue', () => {
    it('should validate values within range', () => {
      expect(validateSensorValue(25, 0, 50)).toBe(true);
      expect(validateSensorValue(0, 0, 50)).toBe(true);
      expect(validateSensorValue(50, 0, 50)).toBe(true);
    });

    it('should reject values outside range', () => {
      expect(validateSensorValue(-1, 0, 50)).toBe(false);
      expect(validateSensorValue(51, 0, 50)).toBe(false);
    });

    it('should handle very small values', () => {
      expect(validateSensorValue(1e-40)).toBe(true); // 0 값은 유효
      expect(validateSensorValue(1e-15)).toBe(true);
    });

    it('should reject invalid values', () => {
      expect(validateSensorValue(undefined)).toBe(false);
      expect(validateSensorValue(null as any)).toBe(false);
      expect(validateSensorValue(Infinity)).toBe(false);
      expect(validateSensorValue(-Infinity)).toBe(false);
      expect(validateSensorValue(NaN)).toBe(false);
    });

    it('should use default range when not specified', () => {
      expect(validateSensorValue(25)).toBe(true);
      expect(validateSensorValue(-1001)).toBe(false);
      expect(validateSensorValue(10001)).toBe(false);
    });
  });

  describe('getSensorPrecision', () => {
    it('should return correct precision for each sensor type', () => {
      expect(getSensorPrecision('temperature')).toBe(1);
      expect(getSensorPrecision('humidity')).toBe(1);
      expect(getSensorPrecision('pressure')).toBe(1);
      expect(getSensorPrecision('illumination')).toBe(0);
      expect(getSensorPrecision('pm10')).toBe(1);
      expect(getSensorPrecision('pm25')).toBe(1);
      expect(getSensorPrecision('tvoc')).toBe(1);
      expect(getSensorPrecision('co2')).toBe(0);
      expect(getSensorPrecision('o2')).toBe(1);
      expect(getSensorPrecision('co')).toBe(2);
    });

    it('should return default precision for unknown sensor types', () => {
      expect(getSensorPrecision('unknown')).toBe(1);
      expect(getSensorPrecision('')).toBe(1);
    });
  });
}); 