import { useState, useEffect, useRef } from 'react';
import { getDevices, Device } from '../services/api';

interface UseDevicesReturn {
  devices: Device[];
  loading: boolean;
  error: string | null;
  refetch: () => Promise<void>;
}

const CACHE_DURATION = 5 * 60 * 1000; // 5분

export const useDevices = (): UseDevicesReturn => {
  const [devices, setDevices] = useState<Device[]>([]);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);
  
  const cacheRef = useRef<{
    data: Device[];
    timestamp: number;
  } | null>(null);
  
  const fetchingRef = useRef<Promise<void> | null>(null);

  const fetchDevices = async (): Promise<void> => {
    // 이미 요청 중인 경우 기존 요청을 기다림
    if (fetchingRef.current) {
      await fetchingRef.current;
      return;
    }

    // 캐시가 유효한 경우 캐시된 데이터 사용
    if (cacheRef.current && Date.now() - cacheRef.current.timestamp < CACHE_DURATION) {
      setDevices(cacheRef.current.data);
      setLoading(false);
      setError(null);
      return;
    }

    // 새로운 요청 시작
    fetchingRef.current = (async () => {
      try {
        setLoading(true);
        setError(null);
        
        const data = await getDevices();
        
        // 캐시 업데이트
        cacheRef.current = {
          data,
          timestamp: Date.now()
        };
        
        setDevices(data);
      } catch (err) {
        setError('디바이스 목록을 불러오는 중 오류가 발생했습니다.');
        console.error('Failed to fetch devices:', err);
      } finally {
        setLoading(false);
        fetchingRef.current = null;
      }
    })();

    await fetchingRef.current;
  };

  const refetch = async (): Promise<void> => {
    // 캐시 무효화
    cacheRef.current = null;
    await fetchDevices();
  };

  useEffect(() => {
    fetchDevices();
  }, []);

  return {
    devices,
    loading,
    error,
    refetch
  };
}; 