package models

import (
	"time"
)

// SensorReading 센서 읽기 데이터 모델
type SensorReading struct {
	ID           uint      `json:"id" gorm:"primaryKey"`
	DeviceID     string    `json:"device_id" gorm:"type:varchar(20);not null;index"`
	NodeID       int       `json:"node_id" gorm:"not null"`
	Temperature  float64   `json:"temperature" gorm:"type:decimal(5,2)"`
	Humidity     float64   `json:"humidity" gorm:"type:decimal(5,2)"`
	Longitude    float64   `json:"longitude" gorm:"type:decimal(10,6)"`
	Latitude     float64   `json:"latitude" gorm:"type:decimal(10,6)"`
	// 추가 센서 데이터 필드
	FloatValue        float64 `json:"float_value" gorm:"type:decimal(10,4)"`
	SignedInt32Value  int32   `json:"signed_int32_value"`
	UnsignedInt32Value uint32 `json:"unsigned_int32_value"`
	// 원시 데이터 (디버깅용)
	RawTem            float64 `json:"raw_tem" gorm:"type:decimal(10,4)"`
	RawHum            float64 `json:"raw_hum" gorm:"type:decimal(10,4)"`
	// 환경 센서 데이터
	PM10             float64 `json:"pm10" gorm:"type:decimal(6,2)"`
	PM25             float64 `json:"pm25" gorm:"type:decimal(6,2)"`
	Pressure         float64 `json:"pressure" gorm:"type:decimal(6,2)"`
	Illumination     float64 `json:"illumination" gorm:"type:decimal(8,2)"`
	TVOC             float64 `json:"tvoc" gorm:"type:decimal(6,2)"`
	CO2              float64 `json:"co2" gorm:"type:decimal(6,2)"`
	O2               float64 `json:"o2" gorm:"type:decimal(5,2)"`
	CO               float64 `json:"co" gorm:"type:decimal(6,2)"`
	RecordedTime      time.Time `json:"recorded_time" gorm:"not null"`
	ReceivedTime      time.Time `json:"received_time" gorm:"not null"`
	CreatedAt         time.Time `json:"created_at" gorm:"autoCreateTime"`
}

// Device 디바이스 정보 모델
type Device struct {
	ID          uint      `json:"id" gorm:"primaryKey"`
	DeviceID    string    `json:"device_id" gorm:"type:varchar(20);not null;unique"`
	Name        string    `json:"name" gorm:"type:varchar(100)"`
	Description string    `json:"description" gorm:"type:text"`
	Status      string    `json:"status" gorm:"type:varchar(20);default:'active'"`
	LastSeen    time.Time `json:"last_seen"`
	CreatedAt   time.Time `json:"created_at" gorm:"autoCreateTime"`
	UpdatedAt   time.Time `json:"updated_at" gorm:"autoUpdateTime"`
}

// SensorDataRequest 센서 데이터 수신 요청 구조체
type SensorDataRequest struct {
	DeviceID     string    `json:"device_id" binding:"required"`
	NodeID       int       `json:"node_id" binding:"required"`
	Temperature  float64   `json:"temperature"`
	Humidity     float64   `json:"humidity"`
	Longitude    float64   `json:"longitude"`
	Latitude     float64   `json:"latitude"`
	// 추가 센서 데이터 필드
	FloatValue        float64 `json:"float_value"`
	SignedInt32Value  int32   `json:"signed_int32_value"`
	UnsignedInt32Value uint32 `json:"unsigned_int32_value"`
	// 원시 데이터 (디버깅용)
	RawTem            float64 `json:"raw_tem"`
	RawHum            float64 `json:"raw_hum"`
	// 환경 센서 데이터
	PM10             float64 `json:"pm10"`
	PM25             float64 `json:"pm25"`
	Pressure         float64 `json:"pressure"`
	Illumination     float64 `json:"illumination"`
	TVOC             float64 `json:"tvoc"`
	CO2              float64 `json:"co2"`
	O2               float64 `json:"o2"`
	CO               float64 `json:"co"`
	RecordedTime      string  `json:"recorded_time" binding:"required"`
}

// SensorDataResponse 센서 데이터 응답 구조체
type SensorDataResponse struct {
	Success bool   `json:"success"`
	Message string `json:"message"`
	Data    *SensorReading `json:"data,omitempty"`
}

// ExtendedSensorDataRequest 확장된 센서 데이터 수신 요청 구조체
type ExtendedSensorDataRequest struct {
	DeviceID     string    `json:"device_id" binding:"required"`
	NodeID       int       `json:"node_id" binding:"required"`
	Temperature  float64   `json:"temperature"`
	Humidity     float64   `json:"humidity"`
	Longitude    float64   `json:"longitude"`
	Latitude     float64   `json:"latitude"`
	// 추가 센서 데이터 필드
	FloatValue        float64 `json:"float_value"`
	SignedInt32Value  int32   `json:"signed_int32_value"`
	UnsignedInt32Value uint32 `json:"unsigned_int32_value"`
	// 원시 데이터 (디버깅용)
	RawTem            float64 `json:"raw_tem"`
	RawHum            float64 `json:"raw_hum"`
	// 환경 센서 데이터
	PM10             float64 `json:"pm10"`
	PM25             float64 `json:"pm25"`
	Pressure         float64 `json:"pressure"`
	Illumination     float64 `json:"illumination"`
	TVOC             float64 `json:"tvoc"`
	CO2              float64 `json:"co2"`
	O2               float64 `json:"o2"`
	CO               float64 `json:"co"`
	RecordedTime      string  `json:"recorded_time" binding:"required"`
}

// DeviceListResponse 디바이스 목록 응답 구조체
type DeviceListResponse struct {
	Success bool     `json:"success"`
	Devices []Device `json:"devices"`
}

// HistoryResponse 히스토리 데이터 응답 구조체
type HistoryResponse struct {
	Success bool            `json:"success"`
	Data    []SensorReading `json:"data"`
	Total   int64           `json:"total"`
} 