#!/bin/bash

# 센서 데이터 수집 시스템 배포 스크립트
# 사용법: ./deploy.sh [dev|staging|prod]

set -e

# 색상 정의
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
NC='\033[0m' # No Color

# 로그 함수
log_info() {
    echo -e "${GREEN}[INFO]${NC} $1"
}

log_warn() {
    echo -e "${YELLOW}[WARN]${NC} $1"
}

log_error() {
    echo -e "${RED}[ERROR]${NC} $1"
}

# 환경 설정
ENVIRONMENT=${1:-dev}
COMPOSE_FILE="docker-compose.yml"

# 환경별 설정
case $ENVIRONMENT in
    "dev")
        log_info "개발 환경으로 배포합니다..."
        export GIN_MODE=debug
        ;;
    "staging")
        log_info "스테이징 환경으로 배포합니다..."
        export GIN_MODE=release
        COMPOSE_FILE="docker-compose.staging.yml"
        ;;
    "prod")
        log_info "프로덕션 환경으로 배포합니다..."
        export GIN_MODE=release
        COMPOSE_FILE="docker-compose.prod.yml"
        ;;
    *)
        log_error "잘못된 환경입니다. 사용법: $0 [dev|staging|prod]"
        exit 1
        ;;
esac

# 배포 전 체크
log_info "배포 전 시스템 체크를 수행합니다..."

# Docker 설치 확인
if ! command -v docker &> /dev/null; then
    log_error "Docker가 설치되어 있지 않습니다."
    exit 1
fi

# Docker Compose 설치 확인
if ! command -v docker-compose &> /dev/null; then
    log_error "Docker Compose가 설치되어 있지 않습니다."
    exit 1
fi

# 포트 사용 확인
PORTS=(8080 3000 5432 6379 8020)
for port in "${PORTS[@]}"; do
    if netstat -tuln | grep ":$port " > /dev/null; then
        log_warn "포트 $port가 이미 사용 중입니다."
    fi
done

# 기존 컨테이너 정리
log_info "기존 컨테이너를 정리합니다..."
docker-compose down --remove-orphans 2>/dev/null || true

# 이미지 빌드
log_info "Docker 이미지를 빌드합니다..."
docker-compose build --no-cache

# 서비스 시작
log_info "서비스를 시작합니다..."
docker-compose up -d

# 헬스체크
log_info "서비스 헬스체크를 수행합니다..."
sleep 10

# Go 서버 헬스체크
if curl -f http://localhost:8080/api/health > /dev/null 2>&1; then
    log_info "Go 서버가 정상적으로 시작되었습니다."
else
    log_error "Go 서버 헬스체크 실패"
    docker-compose logs sensor-server
    exit 1
fi

# PostgreSQL 연결 확인
if docker-compose exec -T postgres pg_isready -U postgres > /dev/null 2>&1; then
    log_info "PostgreSQL이 정상적으로 시작되었습니다."
else
    log_error "PostgreSQL 연결 실패"
    docker-compose logs postgres
    exit 1
fi

# Redis 연결 확인
if docker-compose exec -T redis redis-cli ping > /dev/null 2>&1; then
    log_info "Redis가 정상적으로 시작되었습니다."
else
    log_error "Redis 연결 실패"
    docker-compose logs redis
    exit 1
fi

# 서비스 상태 확인
log_info "모든 서비스 상태를 확인합니다..."
docker-compose ps

# 배포 완료
log_info "배포가 완료되었습니다!"
log_info "웹 대시보드: http://localhost:3000"
log_info "API 서버: http://localhost:8080"
log_info "API 문서: http://localhost:8080/api/health"

# 로그 모니터링 시작
log_info "로그 모니터링을 시작합니다 (Ctrl+C로 종료)..."
docker-compose logs -f 