# 3D 객체인식 평가시스템

[![License: MIT](https://img.shields.io/badge/License-MIT-yellow.svg)](https://opensource.org/licenses/MIT)
[![Python 3.8+](https://img.shields.io/badge/python-3.8+-blue.svg)](https://www.python.org/downloads/)
[![Docker](https://img.shields.io/badge/docker-supported-blue.svg)](https://www.docker.com/)

3D 객체인식 평가시스템은 TRELLIS 모델을 통해 변환된 3D 모델의 객체 인식 정확도를 측정하는 시스템입니다. 다양한 평가 지표를 활용하여 정확하고 신뢰할 수 있는 평가 결과를 제공합니다.

## ✨ 주요 기능

### 🎯 종합 평가 시스템
- **다중 지표 평가**: 2D/3D mAP, Chamfer Distance, EMD, 클래스 정확도
- **가중치 기반 점수**: 사용자 정의 가중치로 맞춤형 평가
- **성능 등급**: A-F 등급으로 직관적인 성능 표시
- **개선된 점수 정규화**: 지표별 특성을 고려한 정규화 알고리즘

### 🎨 3D 렌더링 시스템
- **다각도 렌더링**: 다양한 시점에서의 이미지 생성
- **OSMesa 지원**: 헤드리스 렌더링 환경 지원
- **고품질 렌더링**: 512x512 해상도로 고품질 이미지 생성
- **자동 카메라 설정**: 최적의 카메라 위치 자동 계산

### 📊 평가 지표
- **2D mAP**: 2D 객체 탐지 정확도 측정
- **3D mAP**: 3D 객체 탐지 정확도 측정
- **Chamfer Distance**: 3D 기하학적 유사성 측정
- **EMD (Earth Mover's Distance)**: 3D 분포 유사성 측정
- **클래스 정확도**: 객체 분류 정확도 측정

### 🔧 시스템 기능
- **Docker 지원**: 컨테이너화된 배포 환경
- **자동 테스트**: 단위, 통합, 성능 테스트
- **성능 모니터링**: 메모리 사용량 및 실행 시간 추적
- **상세 로깅**: 평가 과정의 상세한 로그 기록

## 📋 시스템 요구사항

- **Python**: 3.8 이상
- **메모리**: 최소 8GB RAM
- **디스크**: 최소 10GB 여유 공간
- **GPU**: CUDA 11.0 이상 (선택사항, 성능 향상용)

## 🚀 빠른 시작

### 1. 설치

```bash
# 저장소 클론
git clone https://github.com/your-repo/rvtouch3d-evaluation.git
cd rvtouch3d-evaluation

# 가상환경 생성 및 활성화
python -m venv venv
source venv/bin/activate  # Linux/Mac
# 또는
venv\Scripts\activate  # Windows

# 의존성 설치
pip install -r requirements.txt
```

### 2. 기본 사용법

#### Python 코드로 실행
```python
from src.evaluator import Evaluator

# 평가기 초기화
evaluator = Evaluator()

# 모델 평가
result = evaluator.evaluate_model(
    model_path='path/to/your/model.glb',
    reference_path='path/to/reference/image.jpg'
)

# 결과 확인
print(f"종합 점수: {result['comprehensive_score']:.2f}")
print(f"성능 등급: {result['grade']}")
```

#### 명령행으로 실행
```bash
# 기본 실행
python main.py --model path/to/model.glb --reference path/to/reference.jpg

# 출력 디렉토리 지정
python main.py --model model.glb --reference reference.jpg --output results/

# 시각화 포함
python main.py --model model.glb --reference reference.jpg --visualize

# 상세 로그
python main.py --model model.glb --reference reference.jpg --log-level DEBUG
```

### 3. Docker로 실행

```bash
# Docker 이미지 빌드
docker build -t rvtouch3d-evaluation .

# Docker 컨테이너 실행
docker run -v $(pwd)/data:/app/data -v $(pwd)/results:/app/results \
  rvtouch3d-evaluation --model data/model.glb --reference data/reference.jpg
```

## 📚 문서

- [사용자 가이드](docs/user_guide.md) - 상세한 사용법과 예제
- [API 레퍼런스](docs/api_reference.md) - 모든 API 문서
- [개발자 가이드](docs/developer_guide.md) - 개발 및 기여 가이드

## 🧪 테스트

### 자동 테스트 실행

```bash
# 모든 테스트 실행
python tests/automated_tests.py

# 특정 테스트 실행
python -m pytest tests/unit_tests.py -v

# 커버리지 테스트
python tests/test_coverage.py
```

### 테스트 커버리지

```bash
# 커버리지 리포트 생성
python -m pytest tests/ --cov=src --cov-report=html
open htmlcov/index.html
```

## 🔧 설정

시스템 설정은 `config/evaluation_config.py`에서 관리됩니다:

### 평가 가중치 설정
```python
EVALUATION_WEIGHTS = {
    '2d_map': 0.25,          # 2D mAP 가중치 (25%)
    '3d_map': 0.30,          # 3D mAP 가중치 (30%)
    'chamfer_distance': 0.20, # Chamfer Distance 가중치 (20%)
    'emd': 0.15,             # EMD 가중치 (15%)
    'class_accuracy': 0.10   # 클래스 정확도 가중치 (10%)
}
```

**가중치 설명:**

각 지표는 3D 모델의 서로 다른 측면을 평가하며, 가중치는 전체 점수에서 각 지표가 차지하는 비중을 나타냅니다. 총 가중치는 1.0(100%)입니다.

- **2D mAP (0.25 = 25%)**: 
  - **의미**: 2D 이미지에서 객체를 얼마나 정확하게 찾아내는지 측정
  - **예시**: 의자를 찍은 사진에서 "의자가 여기 있다"고 정확히 표시하는 능력
  - **왜 25%인가**: 2D는 3D 모델의 한 면만 보여주므로 상대적으로 낮은 비중
  - **영향**: 2D 성능이 좋지 않아도 다른 지표로 보완 가능

- **3D mAP (0.30 = 30%)**: 
  - **의미**: 3D 공간에서 객체의 위치와 크기를 얼마나 정확하게 파악하는지 측정
  - **예시**: 의자의 3D 위치, 크기, 방향을 정확히 인식하는 능력
  - **왜 30%인가**: 3D 모델의 핵심 기능이므로 가장 높은 비중
  - **영향**: 3D 성능이 전체 평가에 가장 큰 영향을 미침

- **Chamfer Distance (0.20 = 20%)**: 
  - **의미**: 3D 모델의 모양이 실제 객체와 얼마나 비슷한지 측정
  - **예시**: 의자 모델이 실제 의자와 형태적으로 얼마나 유사한지 비교
  - **왜 20%인가**: 모양의 정확성은 중요한 평가 요소
  - **영향**: 모델이 실제 객체와 비슷할수록 높은 점수

- **EMD (0.15 = 15%)**: 
  - **의미**: 3D 모델의 점 분포가 실제 객체의 점 분포와 얼마나 일치하는지 측정
  - **예시**: 의자 표면의 점들이 실제 의자의 표면 분포와 얼마나 비슷한지
  - **왜 15%인가**: 세밀한 분포 비교이지만 상대적으로 낮은 비중
  - **영향**: 모델의 세부적인 품질을 평가

- **클래스 정확도 (0.10 = 10%)**: 
  - **의미**: 3D 모델이 어떤 종류의 객체인지 얼마나 정확하게 분류하는지 측정
  - **예시**: "이것은 의자다", "이것은 테이블이다"라고 정확히 구분하는 능력
  - **왜 10%인가**: 분류는 상대적으로 단순한 작업이므로 가장 낮은 비중
  - **영향**: 기본적인 인식 능력을 평가하지만 전체 점수에는 적은 영향

**가중치 조정 가이드:**
- 3D 성능을 더 중시하고 싶다면 → 3D mAP 가중치를 높이세요
- 모양의 정확성을 더 중요하게 생각한다면 → Chamfer Distance 가중치를 높이세요
- 2D 성능도 중요하다면 → 2D mAP 가중치를 높이세요
- 모든 가중치의 합은 1.0이 되도록 조정해야 합니다

### 평가 임계값 설정
```python
EVALUATION_THRESHOLDS = {
    '2d_map': 0.8,           # 2D mAP 임계값 (80% 이상이면 우수)
    '3d_map': 0.7,           # 3D mAP 임계값 (70% 이상이면 우수)
    'chamfer_distance': 0.3, # Chamfer Distance 임계값 (0.3 이하면 우수)
    'emd': 0.3,              # EMD 임계값 (0.3 이하면 우수)
    'class_accuracy': 0.8    # 클래스 정확도 임계값 (80% 이상이면 우수)
}
```

**임계값 설명:**

임계값은 각 지표에서 "우수한 성능"으로 간주되는 기준점입니다. 이 값들을 기준으로 점수가 정규화됩니다.

- **2D mAP (0.8 = 80%)**: 
  - **의미**: 2D 이미지에서 객체를 80% 이상 정확하게 찾아내면 우수
  - **예시**: 100개의 의자 중 80개 이상을 정확히 찾아내면 우수한 성능
  - **점수 계산**: 80% = 70점, 90% = 85점, 100% = 100점
  - **조정 가이드**: 더 엄격하게 평가하려면 0.9로, 관대하게 평가하려면 0.7로 설정

- **3D mAP (0.7 = 70%)**: 
  - **의미**: 3D 공간에서 객체를 70% 이상 정확하게 인식하면 우수
  - **예시**: 3D 공간에서 의자의 위치, 크기, 방향을 70% 이상 정확히 파악하면 우수
  - **점수 계산**: 70% = 70점, 80% = 85점, 90% = 95점
  - **조정 가이드**: 3D는 2D보다 어려우므로 상대적으로 낮은 임계값 사용

- **Chamfer Distance (0.3)**: 
  - **의미**: 모델과 실제 객체 간의 평균 거리가 0.3 이하면 우수
  - **예시**: 의자 모델의 각 점이 실제 의자와 평균 0.3 단위 이하로 떨어져 있으면 우수
  - **점수 계산**: 0.1 = 95점, 0.3 = 70점, 0.5 = 50점
  - **조정 가이드**: 더 정밀한 모양을 원한다면 0.2로, 관대하게 평가하려면 0.5로 설정

- **EMD (0.3)**: 
  - **의미**: 점 분포의 유사성이 0.3 이하면 우수
  - **예시**: 의자 표면의 점 분포가 실제 의자와 0.3 이하의 차이를 보이면 우수
  - **점수 계산**: 0.1 = 95점, 0.3 = 70점, 0.5 = 50점
  - **조정 가이드**: 세밀한 분포를 중시한다면 0.2로, 관대하게 평가하려면 0.5로 설정

- **클래스 정확도 (0.8 = 80%)**: 
  - **의미**: 객체를 80% 이상 정확하게 분류하면 우수
  - **예시**: 100개의 객체 중 80개 이상을 올바른 클래스(의자, 테이블 등)로 분류하면 우수
  - **점수 계산**: 80% = 70점, 90% = 85점, 100% = 100점
  - **조정 가이드**: 분류 정확도를 더 중시한다면 0.9로, 관대하게 평가하려면 0.7로 설정

**임계값 조정 팁:**
- **더 엄격한 평가**: 모든 임계값을 10-20% 높이세요
- **더 관대한 평가**: 모든 임계값을 10-20% 낮추세요
- **특정 지표 중시**: 해당 지표의 임계값만 조정하세요
- **실제 성능 확인**: 여러 모델을 테스트하여 적절한 임계값을 찾으세요

### 렌더링 설정
```python
RENDERING_CONFIG = {
    'num_views': 36,         # 렌더링할 뷰의 개수 (36개 방향)
    'image_size': (512, 512), # 렌더링 이미지 크기 (가로x세로)
    'camera_distance': 2.0,  # 카메라와 객체 간 거리
    'camera_elevation': 30.0, # 카메라 고도각 (도)
    'num_views_for_evaluation': 8 # 평가용 뷰 개수 (성능 최적화)
}
```

**렌더링 파라미터 설명:**
- **num_views (36)**: 객체를 36개 방향에서 렌더링하여 다양한 시점의 이미지를 생성합니다.
- **image_size (512x512)**: 각 렌더링 이미지의 해상도입니다. 높을수록 정확하지만 처리 시간이 증가합니다.
- **camera_distance (2.0)**: 카메라와 객체 중심점 간의 거리입니다. 객체가 화면에 적절히 보이도록 조정됩니다.
- **camera_elevation (30.0)**: 카메라의 수직 각도입니다. 30도는 객체를 위에서 내려다보는 각도입니다.
- **num_views_for_evaluation (8)**: 실제 평가에 사용할 뷰의 개수입니다. 성능 최적화를 위해 전체 뷰 중 일부만 사용합니다.

### 점군 생성 설정
```python
POINTCLOUD_CONFIG = {
    'num_points': 10000,     # 기본 점군 점의 개수
    'num_points_emd': 5000,  # EMD 계산용 점의 개수
    'num_points_chamfer': 10000 # Chamfer Distance 계산용 점의 개수
}
```

**점군 파라미터 설명:**
- **num_points (10000)**: 3D 모델에서 생성할 기본 점군의 점 개수입니다.
- **num_points_emd (5000)**: EMD 계산 시 사용할 점의 개수입니다. EMD는 계산 복잡도가 높아 적은 점을 사용합니다.
- **num_points_chamfer (10000)**: Chamfer Distance 계산 시 사용할 점의 개수입니다.

### 성능 등급 기준
```python
GRADE_THRESHOLDS = {
    'A': 90.0,               # A등급: 90점 이상 (우수)
    'B': 80.0,               # B등급: 80점 이상 (양호)
    'C': 70.0,               # C등급: 70점 이상 (보통)
    'D': 60.0,               # D등급: 60점 이상 (미흡)
    'F': 0.0                 # F등급: 60점 미만 (부족)
}
```

**등급 기준 설명:**
- **A등급 (90점 이상)**: 우수한 성능으로 모든 지표에서 높은 점수를 받은 모델
- **B등급 (80-89점)**: 양호한 성능으로 대부분의 지표에서 좋은 점수를 받은 모델
- **C등급 (70-79점)**: 보통 성능으로 일부 지표에서 개선이 필요한 모델
- **D등급 (60-69점)**: 미흡한 성능으로 여러 지표에서 개선이 필요한 모델
- **F등급 (60점 미만)**: 부족한 성능으로 전반적인 개선이 필요한 모델

### 클래스 분류 설정
```python
CLASSIFICATION_CONFIG = {
    'num_classes': 10,       # 예상 클래스 개수
    'class_names': [         # 클래스 이름 리스트
        'chair', 'table', 'sofa', 'bed', 'desk',
        'bookshelf', 'lamp', 'cabinet', 'door', 'window'
    ],
    'clustering_enabled': True, # 클러스터링 기반 분류 사용 여부
    'num_clusters': 5        # 클러스터 개수
}
```

**클래스 분류 파라미터 설명:**
- **num_classes (10)**: 시스템이 인식할 수 있는 객체 클래스의 총 개수입니다.
- **class_names**: 인식 가능한 객체 클래스들의 이름 리스트입니다. 가구류 중심으로 구성되어 있습니다.
- **clustering_enabled (True)**: 클러스터링 기반 분류 방법을 사용할지 여부입니다.
- **num_clusters (5)**: 클러스터링 시 사용할 클러스터의 개수입니다.

### IoU 임계값 설정
```python
IOU_THRESHOLDS = {
    '2d_iou': [0.5, 0.75],   # 2D IoU 임계값 리스트
    '3d_iou': [0.5, 0.7]     # 3D IoU 임계값 리스트
}
```

**IoU 임계값 설명:**
- **2d_iou [0.5, 0.75]**: 2D 객체 탐지에서 사용하는 IoU 임계값들입니다. 0.5와 0.75에서 mAP를 계산합니다.
- **3d_iou [0.5, 0.7]**: 3D 객체 탐지에서 사용하는 IoU 임계값들입니다. 3D에서는 일반적으로 더 낮은 임계값을 사용합니다.

### 출력 설정
```python
OUTPUT_CONFIG = {
    'save_images': True,     # 렌더링된 이미지 저장 여부
    'save_pointclouds': False, # 점군 데이터 저장 여부
    'generate_report': True, # 리포트 생성 여부
    'report_format': 'html', # 리포트 형식 ('html', 'json', 'txt')
    'visualize_results': True # 결과 시각화 여부
}
```

**출력 설정 설명:**
- **save_images (True)**: 렌더링된 이미지들을 파일로 저장할지 여부입니다.
- **save_pointclouds (False)**: 생성된 점군 데이터를 파일로 저장할지 여부입니다. 기본적으로 저장하지 않습니다.
- **generate_report (True)**: 평가 결과 리포트를 생성할지 여부입니다.
- **report_format ('html')**: 생성할 리포트의 형식입니다. HTML, JSON, 텍스트 중 선택할 수 있습니다.
- **visualize_results (True)**: 평가 결과를 시각화할지 여부입니다.

### 성능 최적화 설정
```python
PERFORMANCE_CONFIG = {
    'parallel_processing': True, # 병렬 처리 사용 여부
    'max_workers': 4,        # 최대 워커 수
    'memory_limit_gb': 8,    # 메모리 제한 (GB)
    'cache_enabled': True,   # 캐시 사용 여부
    'cache_size_mb': 100     # 캐시 크기 (MB)
}
```

**성능 최적화 파라미터 설명:**
- **parallel_processing (True)**: 병렬 처리를 사용하여 평가 속도를 향상시킬지 여부입니다.
- **max_workers (4)**: 병렬 처리 시 사용할 최대 워커(스레드) 수입니다.
- **memory_limit_gb (8)**: 시스템이 사용할 수 있는 최대 메모리 제한입니다.
- **cache_enabled (True)**: 중간 결과를 캐시하여 반복 계산을 줄일지 여부입니다.
- **cache_size_mb (100)**: 캐시에 사용할 메모리 크기입니다.

### 검증 설정
```python
VALIDATION_CONFIG = {
    'validate_input_files': True, # 입력 파일 검증 여부
    'check_file_formats': True,   # 파일 형식 검증 여부
    'min_file_size_kb': 1,        # 최소 파일 크기 (KB)
    'max_file_size_mb': 100       # 최대 파일 크기 (MB)
}
```

**검증 설정 설명:**
- **validate_input_files (True)**: 입력 파일들의 유효성을 검증할지 여부입니다.
- **check_file_formats (True)**: 파일 형식이 지원되는지 검증할지 여부입니다.
- **min_file_size_kb (1)**: 허용되는 최소 파일 크기입니다. 너무 작은 파일은 오류일 가능성이 높습니다.
- **max_file_size_mb (100)**: 허용되는 최대 파일 크기입니다. 메모리 사용량을 제한하기 위한 설정입니다.

## 🏗️ 아키텍처

```
src/
├── evaluator.py              # 메인 평가기
├── data_loader.py            # 데이터 로더
├── renderer.py               # 3D 렌더러
├── visualizer.py             # 결과 시각화
├── metrics/                  # 평가 지표
│   ├── map_2d.py            # 2D mAP 계산기
│   ├── map_3d.py            # 3D mAP 계산기
│   ├── chamfer_distance.py  # Chamfer Distance 계산기
│   ├── emd.py               # EMD 계산기
│   ├── class_accuracy.py    # 클래스 정확도 계산기
│   └── accuracy_calculator.py # 정확도 계산기
├── utils/                    # 유틸리티
│   ├── reference_extractor.py # 참조 데이터 추출기
│   ├── exception_handler.py   # 예외 처리기
│   ├── normalization_validator.py # 정규화 검증기
│   ├── performance_monitor.py # 성능 모니터링
│   ├── logging_utils.py      # 로깅 유틸리티
│   ├── config_validator.py   # 설정 검증기
│   └── system_monitor.py     # 시스템 모니터링
├── config/
│   └── evaluation_config.py  # 평가 설정
├── tests/                    # 테스트
│   ├── automated_tests.py    # 자동화 테스트
│   ├── test_coverage.py      # 커버리지 테스트
│   ├── test_metrics.py       # 지표 테스트
│   └── test_evaluator.py     # 평가기 테스트
└── main.py                   # 메인 실행 파일
```

## 🚀 자동화 및 테스트

### 자동화된 테스트 시스템
- **단위 테스트**: 개별 모듈별 기능 테스트
- **통합 테스트**: 전체 시스템 통합 테스트
- **성능 테스트**: 메모리 사용량 및 실행 시간 테스트
- **커버리지 테스트**: 코드 커버리지 측정

### Docker 지원
- **컨테이너화**: 완전한 Docker 환경 지원
- **OSMesa 렌더링**: 헤드리스 렌더링 환경
- **의존성 관리**: 모든 필요한 라이브러리 포함

## 📊 성능 특징

### 평가 지표
- **2D mAP**: 2D 객체 탐지 정확도 (0.0-1.0)
- **3D mAP**: 3D 객체 탐지 정확도 (0.0-1.0)
- **Chamfer Distance**: 3D 기하학적 유사성 (낮을수록 좋음)
- **EMD**: 3D 분포 유사성 (낮을수록 좋음)
- **클래스 정확도**: 객체 분류 정확도 (0.0-1.0)

### 성능 등급 기준
- **A등급**: 90점 이상 (우수)
- **B등급**: 80점 이상 (양호)
- **C등급**: 70점 이상 (보통)
- **D등급**: 60점 이상 (미흡)
- **F등급**: 60점 미만 (부족)

### 시스템 요구사항
- **평가 시간**: 일반적으로 1-5분 (모델 복잡도에 따라)
- **메모리 사용량**: 4-8GB (모델 크기에 따라)
- **지원 형식**: GLB, OBJ, PLY 등 3D 모델 형식

## 🤝 기여하기

1. Fork the Project
2. Create your Feature Branch (`git checkout -b feature/AmazingFeature`)
3. Commit your Changes (`git commit -m 'Add some AmazingFeature'`)
4. Push to the Branch (`git push origin feature/AmazingFeature`)
5. Open a Pull Request

## 📝 라이선스

이 프로젝트는 MIT 라이선스 하에 배포됩니다. 자세한 내용은 [LICENSE](LICENSE) 파일을 참조하세요.

## 🙏 감사의 말

- [Open3D](https://github.com/isl-org/Open3D) - 3D 데이터 처리
- [Trimesh](https://github.com/mikedh/trimesh) - 3D 메시 처리
- [Ultralytics](https://github.com/ultralytics/ultralytics) - YOLO 모델
- [OpenCV](https://opencv.org/) - 컴퓨터 비전
- [PyTorch](https://pytorch.org/) - 딥러닝 프레임워크
- [NumPy](https://numpy.org/) - 수치 계산
- [Matplotlib](https://matplotlib.org/) - 시각화

## 📞 지원

- **GitHub Issues**: [프로젝트 저장소](https://github.com/your-repo/rvtouch3d-evaluation/issues)
- **이메일**: support@example.com
- **문서**: [프로젝트 문서](https://docs.example.com)

## 🔄 업데이트 로그

### v1.0.0 (2024-01-15)
- 🎯 기본 3D 객체인식 평가 시스템
- 📊 2D/3D mAP, Chamfer Distance, EMD, 클래스 정확도 계산
- 🎨 다각도 3D 렌더링 시스템
- 🐳 Docker 지원 및 OSMesa 헤드리스 렌더링
- 🔧 자동화된 테스트 시스템
- 📚 상세한 문서화 및 API 레퍼런스
- 🎛️ 설정 가능한 평가 가중치 및 임계값
- 📈 성능 모니터링 및 상세 로깅

---

**3D 객체인식 평가시스템**으로 더 정확하고 신뢰할 수 있는 3D 모델 평가를 경험해보세요! 🚀