# 3D 객체인식 평가시스템 사용자 가이드

## 목차

1. [시스템 개요](#시스템-개요)
2. [설치 및 설정](#설치-및-설정)
3. [기본 사용법](#기본-사용법)
4. [고급 기능](#고급-기능)
5. [문제 해결](#문제-해결)
6. [FAQ](#faq)

## 시스템 개요

3D 객체인식 평가시스템은 3D 모델의 성능을 종합적으로 평가하는 시스템입니다. 다음과 같은 주요 기능을 제공합니다:

- **종합 평가**: 2D/3D mAP, Chamfer Distance, EMD, 클래스 정확도 등 다양한 지표로 평가
- **Empir3D 프레임워크**: 해상도, 정확도, 커버리지, 아티팩트 점수 등 새로운 평가 지표
- **DeepEMD**: Transformer 기반 EMD 계산으로 성능 향상
- **SceneDreamer**: 3D 장면 생성 및 다각도 렌더링
- **자동화**: CI/CD 파이프라인과 자동 테스트 시스템

## 설치 및 설정

### 시스템 요구사항

- Python 3.8 이상
- CUDA 11.0 이상 (GPU 사용 시)
- 최소 8GB RAM
- 최소 10GB 디스크 공간

### 설치 방법

1. **저장소 클론**
```bash
git clone https://github.com/your-repo/rvtouch3d-evaluation.git
cd rvtouch3d-evaluation
```

2. **가상환경 생성**
```bash
python -m venv venv
source venv/bin/activate  # Linux/Mac
# 또는
venv\Scripts\activate  # Windows
```

3. **의존성 설치**
```bash
pip install -r requirements.txt
```

4. **YOLO 모델 설치 (선택사항)**
```bash
pip install ultralytics
python -c "from ultralytics import YOLO; YOLO('yolov8n.pt')"
```

### 설정 파일

시스템 설정은 `config/evaluation_config.py`에서 관리됩니다. 주요 설정 항목:

```python
# 평가 가중치
EVALUATION_WEIGHTS = {
    '2d_map': 0.25,
    '3d_map': 0.30,
    'chamfer_distance': 0.20,
    'emd': 0.15,
    'class_accuracy': 0.10
}

# 렌더링 설정
RENDERING_CONFIG = {
    'num_views': 36,
    'image_size': (512, 512),
    'camera_distance': 2.0,
    'camera_elevation': 30.0
}
```

## 기본 사용법

### 1. 간단한 평가

```python
from src.evaluator import Evaluator

# 평가기 초기화
evaluator = Evaluator()

# 모델 평가
result = evaluator.evaluate_model(
    model_path='path/to/your/model.glb',
    reference_path='path/to/reference/image.jpg'
)

# 결과 확인
print(f"종합 점수: {result['comprehensive_score']:.2f}")
print(f"성능 등급: {result['grade']}")
print(f"개별 지표:")
for metric, value in result['metrics'].items():
    print(f"  {metric}: {value:.4f}")
```

### 2. Empir3D 평가

```python
from src.metrics.empir3d_evaluator import Empir3DEvaluator

# Empir3D 평가기 초기화
evaluator = Empir3DEvaluator()

# 3D 모델 데이터 준비
model_3d = {
    'vertices': [[0, 0, 0], [1, 0, 0], [1, 1, 0], [0, 1, 0]],
    'faces': [[0, 1, 2], [0, 2, 3]],
    'center': [0.5, 0.5, 0.0],
    'scale': [1.0, 1.0, 0.0]
}

ground_truth_3d = {
    'vertices': [[0, 0, 0], [1, 0, 0], [1, 1, 0], [0, 1, 0]],
    'faces': [[0, 1, 2], [0, 2, 3]],
    'center': [0.5, 0.5, 0.0],
    'scale': [1.0, 1.0, 0.0]
}

# Empir3D 평가
result = evaluator.evaluate_3d_model(model_3d, ground_truth_3d)

print(f"Empir3D 점수: {result['empir3d_score']:.2f}")
print(f"해상도: {result['resolution']['score']:.2f}")
print(f"정확도: {result['accuracy']['score']:.2f}")
print(f"커버리지: {result['coverage']['score']:.2f}")
print(f"아티팩트: {result['artifact']['score']:.2f}")
```

### 3. 렌더링

```python
from src.renderer.improved_renderer import ImprovedRenderer

# 렌더러 초기화
renderer = ImprovedRenderer()

# 3D 모델 렌더링
rendered_images = renderer.render_multiple_views(model_3d, num_views=8)

print(f"생성된 이미지 수: {len(rendered_images)}")
```

## 고급 기능

### 1. 커스텀 설정

```python
from src.evaluator import Evaluator

# 커스텀 설정
custom_config = {
    'weights': {
        '2d_map': 0.3,
        '3d_map': 0.4,
        'chamfer_distance': 0.2,
        'emd': 0.1,
        'class_accuracy': 0.0
    },
    'rendering': {
        'num_views': 16,
        'image_size': (256, 256),
        'camera_distance': 1.5,
        'camera_elevation': 45.0
    },
    'grade_thresholds': {
        'A': 95.0,
        'B': 85.0,
        'C': 75.0,
        'D': 65.0,
        'F': 0.0
    }
}

# 커스텀 설정으로 평가기 초기화
evaluator = Evaluator(config=custom_config)
```

### 2. DeepEMD 사용

```python
from src.metrics.deep_emd_calculator import DeepEMDCalculator

# DeepEMD 계산기 초기화
calculator = DeepEMDCalculator()

# DeepEMD 계산
emd_value = calculator.calculate_deep_emd(model_3d, ground_truth_3d)

print(f"DeepEMD 값: {emd_value:.4f}")
```

### 3. SceneDreamer 사용

```python
from src.generators.scene_dreamer import SceneDreamer
import numpy as np

# SceneDreamer 초기화
dreamer = SceneDreamer()

# 입력 이미지 준비
input_image = np.random.randint(0, 255, (512, 512, 3), dtype=np.uint8)

# 3D 장면 생성
result = dreamer.generate_3d_scene(input_image, num_views=8)

print(f"생성된 3D 모델 정점 수: {len(result['3d_model']['vertices'])}")
print(f"생성된 뷰 수: {len(result['multi_views'])}")
```

### 4. 장면 탐지

```python
from src.integration.scene_detection import SceneDetectionSystem

# 장면 탐지 시스템 초기화
detector = SceneDetectionSystem()

# 장면 데이터 준비
scene_data = np.random.randint(0, 255, (512, 512, 3), dtype=np.uint8)

# 객체 탐지
result = detector.detect_scene_objects(scene_data)

print(f"탐지된 객체 수: {result['metadata']['num_objects']}")
print(f"장면 복잡도: {result['metadata']['scene_complexity']}")
```

### 5. 배치 처리

```python
import os
from src.evaluator import Evaluator

# 평가기 초기화
evaluator = Evaluator()

# 배치 처리
model_dir = 'path/to/models'
reference_dir = 'path/to/references'

results = []
for model_file in os.listdir(model_dir):
    if model_file.endswith('.glb'):
        model_path = os.path.join(model_dir, model_file)
        reference_path = os.path.join(reference_dir, model_file.replace('.glb', '.jpg'))
        
        if os.path.exists(reference_path):
            result = evaluator.evaluate_model(model_path, reference_path)
            results.append({
                'model': model_file,
                'score': result['comprehensive_score'],
                'grade': result['grade']
            })

# 결과 정렬
results.sort(key=lambda x: x['score'], reverse=True)

# 결과 출력
for result in results:
    print(f"{result['model']}: {result['score']:.2f} ({result['grade']})")
```

## 문제 해결

### 일반적인 문제

#### 1. YOLO 모델 로드 실패
```
ImportError: No module named 'ultralytics'
```
**해결방법**: YOLO 모델 설치
```bash
pip install ultralytics
```

#### 2. Open3D 렌더링 실패
```
ImportError: No module named 'open3d'
```
**해결방법**: Open3D 설치
```bash
pip install open3d
```

#### 3. 메모리 부족 오류
```
MemoryError: Unable to allocate array
```
**해결방법**: 
- 배치 크기 줄이기
- 메모리 사용량 모니터링
- 스트리밍 처리 사용

#### 4. CUDA 오류
```
RuntimeError: CUDA out of memory
```
**해결방법**:
- GPU 메모리 정리
- 배치 크기 줄이기
- CPU 모드 사용

### 로그 확인

시스템은 상세한 로그를 제공합니다:

```python
import logging

# 로그 레벨 설정
logging.basicConfig(level=logging.DEBUG)

# 로그 파일 확인
tail -f results/evaluation.log
```

### 성능 최적화

#### 1. 메모리 최적화
```python
# 메모리 사용량 모니터링
import psutil
import os

process = psutil.Process(os.getpid())
memory_usage = process.memory_info().rss / 1024 / 1024  # MB
print(f"메모리 사용량: {memory_usage:.2f} MB")
```

#### 2. GPU 사용
```python
import torch

# GPU 사용 가능 여부 확인
if torch.cuda.is_available():
    print(f"GPU 사용 가능: {torch.cuda.get_device_name()}")
    device = torch.device('cuda')
else:
    print("GPU 사용 불가능, CPU 사용")
    device = torch.device('cpu')
```

#### 3. 병렬 처리
```python
from multiprocessing import Pool
import functools

# 병렬 처리 함수
def evaluate_model_parallel(args):
    model_path, reference_path = args
    evaluator = Evaluator()
    return evaluator.evaluate_model(model_path, reference_path)

# 병렬 실행
with Pool(processes=4) as pool:
    results = pool.map(evaluate_model_parallel, model_pairs)
```

## FAQ

### Q: 시스템의 최소 요구사항은 무엇인가요?
A: Python 3.8 이상, 8GB RAM, 10GB 디스크 공간이 필요합니다. GPU 사용 시 CUDA 11.0 이상이 권장됩니다.

### Q: 어떤 3D 모델 형식을 지원하나요?
A: 현재 GLB 형식을 지원합니다. 다른 형식은 변환 후 사용하세요.

### Q: 평가 시간은 얼마나 걸리나요?
A: 모델 크기와 설정에 따라 다릅니다. 일반적으로 1-5분 정도 소요됩니다.

### Q: 커스텀 평가 지표를 추가할 수 있나요?
A: 네, `src/metrics/` 디렉토리에 새로운 지표를 추가할 수 있습니다.

### Q: GPU 없이도 사용할 수 있나요?
A: 네, CPU 모드로도 사용 가능합니다. 다만 성능이 느릴 수 있습니다.

### Q: 평가 결과를 저장할 수 있나요?
A: 네, JSON, HTML, 텍스트 형식으로 결과를 저장할 수 있습니다.

### Q: 배치 처리는 어떻게 하나요?
A: 위의 "배치 처리" 섹션을 참고하세요. 여러 모델을 한 번에 평가할 수 있습니다.

### Q: 오류가 발생하면 어떻게 하나요?
A: 로그 파일을 확인하고, 위의 "문제 해결" 섹션을 참고하세요. 문제가 지속되면 GitHub Issues에 문의하세요.

### Q: 시스템을 확장할 수 있나요?
A: 네, 모듈화된 구조로 설계되어 있어 새로운 기능을 쉽게 추가할 수 있습니다.

### Q: 상업적 사용이 가능한가요?
A: MIT 라이선스 하에 배포되므로 상업적 사용이 가능합니다.

## 추가 리소스

- [API 레퍼런스](api_reference.md)
- [개발자 가이드](developer_guide.md)
- [GitHub 저장소](https://github.com/your-repo/rvtouch3d-evaluation)
- [이슈 트래커](https://github.com/your-repo/rvtouch3d-evaluation/issues)
- [위키](https://github.com/your-repo/rvtouch3d-evaluation/wiki)

## 지원

문제가 발생하거나 질문이 있으시면 다음을 통해 문의하세요:

- GitHub Issues: [프로젝트 저장소](https://github.com/your-repo/rvtouch3d-evaluation)
- 이메일: support@example.com
- 문서: [프로젝트 문서](https://docs.example.com)
