# 3D 객체인식 평가시스템 API 레퍼런스

## 개요

3D 객체인식 평가시스템은 3D 모델의 성능을 종합적으로 평가하는 시스템입니다. 이 문서는 시스템의 주요 API와 사용법을 설명합니다.

## 주요 컴포넌트

### 1. Evaluator (평가기)

#### 클래스: `Evaluator`

3D 모델을 종합적으로 평가하는 메인 클래스입니다.

```python
from src.evaluator import Evaluator

# 초기화
evaluator = Evaluator(config=None)

# 모델 평가
result = evaluator.evaluate_model(model_path, reference_path)
```

#### 메서드

##### `__init__(config: Dict = None)`
- **설명**: 평가기 초기화
- **매개변수**:
  - `config` (Dict, 선택): 평가 설정
- **반환값**: None

##### `evaluate_model(model_path: str, reference_path: str) -> Dict`
- **설명**: 3D 모델을 종합적으로 평가
- **매개변수**:
  - `model_path` (str): 3D 모델 파일 경로
  - `reference_path` (str): 참조 이미지 파일 경로
- **반환값**: Dict - 평가 결과
  - `comprehensive_score` (float): 종합 점수
  - `grade` (str): 성능 등급 (A, B, C, D, F)
  - `metrics` (Dict): 개별 지표 결과
  - `score_details` (Dict): 점수 세부사항

##### `calculate_comprehensive_score(metrics: Dict) -> Tuple[float, Dict]`
- **설명**: 가중치 기반 종합 점수 계산
- **매개변수**:
  - `metrics` (Dict): 개별 평가 지표 결과
- **반환값**: Tuple[float, Dict] - (종합 점수, 점수 세부사항)

##### `determine_grade(score: float) -> str`
- **설명**: 점수에 따라 성능 등급 결정
- **매개변수**:
  - `score` (float): 종합 점수 (0-100)
- **반환값**: str - 성능 등급

### 2. Empir3D Evaluator (Empir3D 평가기)

#### 클래스: `Empir3DEvaluator`

새로운 3D 평가 지표를 사용하는 평가기입니다.

```python
from src.metrics.empir3d_evaluator import Empir3DEvaluator

# 초기화
evaluator = Empir3DEvaluator(config=None)

# 3D 모델 평가
result = evaluator.evaluate_3d_model(model_3d, ground_truth_3d)
```

#### 메서드

##### `evaluate_3d_model(model_3d: Dict, ground_truth_3d: Dict) -> Dict`
- **설명**: 3D 모델을 Empir3D 프레임워크로 평가
- **매개변수**:
  - `model_3d` (Dict): 3D 모델 데이터
  - `ground_truth_3d` (Dict): Ground Truth 데이터
- **반환값**: Dict - Empir3D 평가 결과
  - `empir3d_score` (float): Empir3D 종합 점수
  - `resolution` (Dict): 해상도 점수
  - `accuracy` (Dict): 정확도 점수
  - `coverage` (Dict): 커버리지 점수
  - `artifact` (Dict): 아티팩트 점수

### 3. Reference Data Extractor (참조 데이터 추출기)

#### 클래스: `ReferenceDataExtractor`

참조 이미지에서 Ground Truth 데이터를 추출합니다.

```python
from src.utils.reference_extractor import ReferenceDataExtractor

# 초기화
extractor = ReferenceDataExtractor()

# Ground Truth 생성
ground_truth = extractor.create_unified_ground_truth(reference_image_path)
```

#### 메서드

##### `extract_reference_data(image_path: str) -> Dict`
- **설명**: 참조 이미지에서 객체 정보 추출
- **매개변수**:
  - `image_path` (str): 참조 이미지 경로
- **반환값**: Dict - 추출된 객체 정보

##### `create_unified_ground_truth(reference_image_path: str, model_3d_path: str = None) -> Dict`
- **설명**: 표준화된 Ground Truth 데이터 생성
- **매개변수**:
  - `reference_image_path` (str): 참조 이미지 경로
  - `model_3d_path` (str, 선택): 3D 모델 경로
- **반환값**: Dict - 표준화된 Ground Truth 데이터

### 4. DeepEMD Calculator (DeepEMD 계산기)

#### 클래스: `DeepEMDCalculator`

Transformer 기반 EMD 계산을 수행합니다.

```python
from src.metrics.deep_emd_calculator import DeepEMDCalculator

# 초기화
calculator = DeepEMDCalculator(config=None)

# DeepEMD 계산
emd_value = calculator.calculate_deep_emd(model_3d, ground_truth_3d)
```

#### 메서드

##### `calculate_deep_emd(model_3d: Dict, ground_truth_3d: Dict) -> float`
- **설명**: DeepEMD 계산
- **매개변수**:
  - `model_3d` (Dict): 3D 모델 데이터
  - `ground_truth_3d` (Dict): Ground Truth 데이터
- **반환값**: float - DeepEMD 값

### 5. Improved Renderer (개선된 렌더러)

#### 클래스: `ImprovedRenderer`

고품질 렌더링을 제공합니다.

```python
from src.renderer.improved_renderer import ImprovedRenderer

# 초기화
renderer = ImprovedRenderer(config=None)

# 다각도 뷰 렌더링
rendered_images = renderer.render_multiple_views(model_3d, num_views=8)
```

#### 메서드

##### `render_multiple_views(model_3d: Dict, num_views: Optional[int] = None) -> List[np.ndarray]`
- **설명**: 다각도 뷰 렌더링
- **매개변수**:
  - `model_3d` (Dict): 3D 모델 데이터
  - `num_views` (Optional[int]): 렌더링할 뷰 개수
- **반환값**: List[np.ndarray] - 렌더링된 이미지 리스트

### 6. SceneDreamer (장면 생성기)

#### 클래스: `SceneDreamer`

3D 장면을 생성합니다.

```python
from src.generators.scene_dreamer import SceneDreamer

# 초기화
dreamer = SceneDreamer(config=None)

# 3D 장면 생성
result = dreamer.generate_3d_scene(input_data, num_views=8)
```

#### 메서드

##### `generate_3d_scene(input_data: Union[np.ndarray, Dict], num_views: int = 8) -> Dict`
- **설명**: 3D 장면 생성
- **매개변수**:
  - `input_data` (Union[np.ndarray, Dict]): 입력 데이터
  - `num_views` (int): 생성할 뷰 개수
- **반환값**: Dict - 생성된 3D 장면 데이터

### 7. Scene Detection System (장면 탐지 시스템)

#### 클래스: `SceneDetectionSystem`

복잡한 3D 장면을 처리하고 다중 객체를 탐지합니다.

```python
from src.integration.scene_detection import SceneDetectionSystem

# 초기화
detector = SceneDetectionSystem(config=None)

# 장면 객체 탐지
result = detector.detect_scene_objects(scene_data)
```

#### 메서드

##### `detect_scene_objects(scene_data: Union[np.ndarray, Dict]) -> Dict`
- **설명**: 장면에서 객체 탐지
- **매개변수**:
  - `scene_data` (Union[np.ndarray, Dict]): 장면 데이터
- **반환값**: Dict - 탐지된 객체 정보

## 데이터 구조

### 3D 모델 데이터 구조

```python
model_3d = {
    'vertices': List[List[float]],  # 3D 정점 좌표
    'faces': List[List[int]],       # 면 인덱스
    'center': List[float],          # 중심점
    'scale': List[float]            # 스케일
}
```

### Ground Truth 데이터 구조

```python
ground_truth = {
    'vertices': List[List[float]],     # 3D 정점 좌표
    'faces': List[List[int]],          # 면 인덱스
    'center': List[float],             # 중심점
    'scale': List[float],              # 스케일
    'objects': List[Dict],             # 객체 정보
    'bounding_boxes': List[List[float]], # 바운딩 박스
    'class_labels': List[int],         # 클래스 레이블
    'confidence_scores': List[float],  # 신뢰도 점수
    'metadata': Dict                   # 메타데이터
}
```

### 평가 결과 데이터 구조

```python
evaluation_result = {
    'comprehensive_score': float,      # 종합 점수
    'grade': str,                     # 성능 등급
    'metrics': {                      # 개별 지표
        '2d_map': float,
        '3d_map': float,
        'chamfer_distance': float,
        'emd': float,
        'class_accuracy': float
    },
    'score_details': Dict,            # 점수 세부사항
    'metadata': Dict                  # 메타데이터
}
```

## 설정

### 평가 설정

```python
evaluation_config = {
    'weights': {                      # 가중치
        '2d_map': 0.25,
        '3d_map': 0.30,
        'chamfer_distance': 0.20,
        'emd': 0.15,
        'class_accuracy': 0.10
    },
    'thresholds': {                   # 임계값
        '2d_map': 0.8,
        '3d_map': 0.7,
        'chamfer_distance': 0.1,
        'emd': 0.15,
        'class_accuracy': 0.8
    },
    'rendering': {                    # 렌더링 설정
        'num_views': 36,
        'image_size': (512, 512),
        'camera_distance': 2.0,
        'camera_elevation': 30.0
    },
    'grade_thresholds': {             # 등급 임계값
        'A': 90.0,
        'B': 80.0,
        'C': 70.0,
        'D': 60.0,
        'F': 0.0
    }
}
```

## 사용 예제

### 기본 사용법

```python
from src.evaluator import Evaluator

# 평가기 초기화
evaluator = Evaluator()

# 모델 평가
result = evaluator.evaluate_model(
    model_path='path/to/model.glb',
    reference_path='path/to/reference.jpg'
)

# 결과 출력
print(f"종합 점수: {result['comprehensive_score']:.2f}")
print(f"성능 등급: {result['grade']}")
```

### Empir3D 사용법

```python
from src.metrics.empir3d_evaluator import Empir3DEvaluator

# Empir3D 평가기 초기화
evaluator = Empir3DEvaluator()

# 3D 모델 평가
result = evaluator.evaluate_3d_model(model_3d, ground_truth_3d)

# 결과 출력
print(f"Empir3D 점수: {result['empir3d_score']:.2f}")
print(f"해상도 점수: {result['resolution']['score']:.2f}")
print(f"정확도 점수: {result['accuracy']['score']:.2f}")
```

### 커스텀 설정 사용법

```python
from src.evaluator import Evaluator

# 커스텀 설정
custom_config = {
    'weights': {
        '2d_map': 0.3,
        '3d_map': 0.4,
        'chamfer_distance': 0.2,
        'emd': 0.1,
        'class_accuracy': 0.0
    },
    'rendering': {
        'num_views': 16,
        'image_size': (256, 256)
    }
}

# 커스텀 설정으로 평가기 초기화
evaluator = Evaluator(config=custom_config)

# 모델 평가
result = evaluator.evaluate_model(model_path, reference_path)
```

## 오류 처리

시스템은 견고한 오류 처리를 제공합니다:

- **KeyError**: vertices 키 누락 시 자동으로 대체 데이터 생성
- **MemoryError**: 메모리 부족 시 스트리밍 처리로 전환
- **ValueError**: 잘못된 입력 시 기본값 사용
- **ImportError**: 의존성 누락 시 대체 방법 사용

## 성능 최적화

- **병렬 처리**: 다각도 렌더링 시 병렬 처리 사용
- **메모리 최적화**: 대용량 데이터 스트리밍 처리
- **캐싱**: 계산 결과 캐싱으로 성능 향상
- **GPU 가속**: CUDA 사용 가능 시 GPU 가속

## 로깅

시스템은 상세한 로깅을 제공합니다:

```python
import logging

# 로깅 설정
logging.basicConfig(
    level=logging.INFO,
    format='%(asctime)s - %(name)s - %(levelname)s - %(message)s'
)

# 로그 파일 설정
logging.basicConfig(
    filename='evaluation.log',
    level=logging.DEBUG
)
```

## 테스트

자동화된 테스트 시스템을 제공합니다:

```bash
# 모든 테스트 실행
python tests/automated_tests.py

# 특정 테스트 실행
python -m pytest tests/unit_tests.py -v

# 커버리지 테스트
python -m pytest tests/ --cov=src --cov-report=html
```

## 라이선스

이 프로젝트는 MIT 라이선스 하에 배포됩니다.

## 지원

문제가 발생하거나 질문이 있으시면 다음을 통해 문의하세요:

- GitHub Issues: [프로젝트 저장소](https://github.com/your-repo/rvtouch3d-evaluation)
- 이메일: support@example.com
- 문서: [프로젝트 문서](https://docs.example.com)
