#!/usr/bin/env python3
import os
import re
import json
import nbformat
from nbformat.v4 import new_notebook, new_markdown_cell, new_code_cell

def md_to_ipynb(md_file_path, ipynb_file_path):
    """마크다운 파일을 Jupyter Notebook으로 변환합니다."""
    print(f"변환 중: {md_file_path} -> {ipynb_file_path}")
    
    # 마크다운 파일 읽기
    with open(md_file_path, 'r', encoding='utf-8') as f:
        md_content = f.read()
    
    # 코드 블록과 마크다운 블록 분리
    blocks = []
    current_block = {"type": "markdown", "content": ""}
    
    # 코드 블록 패턴
    code_block_pattern = re.compile(r'```(python|bash|r|sql|)\s*([\s\S]*?)```', re.MULTILINE)
    
    # 마크다운 파일을 파싱하여 코드 블록과 마크다운 블록으로 분리
    last_end = 0
    for match in code_block_pattern.finditer(md_content):
        start, end = match.span()
        
        # 코드 블록 이전의 마크다운 내용 추가
        if start > last_end:
            md_text = md_content[last_end:start]
            if md_text.strip():
                if current_block["type"] == "markdown":
                    current_block["content"] += md_text
                else:
                    # 이전 블록이 코드 블록이었다면 새로운 마크다운 블록 시작
                    if current_block["content"].strip():
                        blocks.append(current_block)
                    current_block = {"type": "markdown", "content": md_text}
        
        # 코드 블록 추가
        code_lang = match.group(1)
        code_content = match.group(2).strip()
        
        # 이전 블록 저장
        if current_block["content"].strip():
            blocks.append(current_block)
        
        # 코드 블록 생성
        current_block = {"type": "code", "content": code_content, "language": code_lang or "python"}
        
        last_end = end
    
    # 마지막 블록 추가
    if last_end < len(md_content):
        md_text = md_content[last_end:]
        if md_text.strip():
            if current_block["type"] == "markdown":
                current_block["content"] += md_text
            else:
                if current_block["content"].strip():
                    blocks.append(current_block)
                current_block = {"type": "markdown", "content": md_text}
    
    # 마지막 블록 저장
    if current_block["content"].strip():
        blocks.append(current_block)
    
    # Jupyter Notebook 생성
    nb = new_notebook()
    
    # 셀 추가
    for block in blocks:
        if block["type"] == "markdown":
            nb.cells.append(new_markdown_cell(block["content"]))
        else:
            nb.cells.append(new_code_cell(block["content"]))
    
    # Jupyter Notebook 저장
    with open(ipynb_file_path, 'w', encoding='utf-8') as f:
        nbformat.write(nb, f)
    
    print(f"변환 완료: {ipynb_file_path}")

def process_directory(input_dir, output_dir=None):
    """디렉토리 내의 모든 마크다운 파일을 Jupyter Notebook으로 변환합니다."""
    if output_dir is None:
        output_dir = input_dir
    
    # 출력 디렉토리가 없으면 생성
    os.makedirs(output_dir, exist_ok=True)
    
    # 디렉토리 내의 모든 마크다운 파일 처리
    for filename in os.listdir(input_dir):
        if filename.endswith('.md'):
            md_file_path = os.path.join(input_dir, filename)
            ipynb_file_path = os.path.join(output_dir, filename.replace('.md', '.ipynb'))
            md_to_ipynb(md_file_path, ipynb_file_path)

if __name__ == "__main__":
    # 변환할 디렉토리 지정
    input_dir = "docs/tutorials/numpy-pandas"
    output_dir = "docs/tutorials/numpy-pandas"
    
    # 변환 실행
    process_directory(input_dir, output_dir)
    print("모든 파일 변환이 완료되었습니다!") 