"""
Part 15: AI 전문가 양성 과정 캡스톤 프로젝트

이 모듈은 AI 전문가 양성 과정의 최종 프로젝트를 위한 기본 구조와 
핵심 기능들을 제공합니다.

주요 기능:
- 프로젝트 템플릿 생성
- 모델 평가 및 성능 측정
- 실무 배포 준비
- 포트폴리오 작성 가이드
"""

import os
import json
import logging
from typing import Dict, List, Any, Optional
from datetime import datetime
import pandas as pd
import numpy as np
from pathlib import Path

# 로깅 설정
logging.basicConfig(level=logging.INFO)
logger = logging.getLogger(__name__)


class CapstoneProject:
    """캡스톤 프로젝트 관리 클래스"""
    
    def __init__(self, project_name: str, project_type: str = "ml"):
        """
        캡스톤 프로젝트 초기화
        
        Args:
            project_name: 프로젝트 이름
            project_type: 프로젝트 타입 ("ml", "nlp", "cv", "rl")
        """
        self.project_name = project_name
        self.project_type = project_type
        self.project_path = Path(f"projects/{project_name}")
        self.config = self._initialize_config()
        
    def _initialize_config(self) -> Dict[str, Any]:
        """프로젝트 설정 초기화"""
        return {
            "project_name": self.project_name,
            "project_type": self.project_type,
            "created_at": datetime.now().isoformat(),
            "version": "1.0.0",
            "status": "initialized"
        }
    
    def create_project_structure(self) -> bool:
        """프로젝트 디렉토리 구조 생성"""
        try:
            # 기본 디렉토리 구조
            directories = [
                "data",
                "models",
                "src",
                "tests",
                "docs",
                "notebooks",
                "scripts",
                "config"
            ]
            
            for directory in directories:
                (self.project_path / directory).mkdir(parents=True, exist_ok=True)
            
            # 설정 파일 생성
            config_file = self.project_path / "config" / "project_config.json"
            with open(config_file, 'w', encoding='utf-8') as f:
                json.dump(self.config, f, indent=2, ensure_ascii=False)
            
            # README 파일 생성
            self._create_readme()
            
            logger.info(f"프로젝트 구조가 성공적으로 생성되었습니다: {self.project_path}")
            return True
            
        except Exception as e:
            logger.error(f"프로젝트 구조 생성 중 오류 발생: {e}")
            return False
    
    def _create_readme(self):
        """README.md 파일 생성"""
        readme_content = f"""# {self.project_name}

## 프로젝트 개요
이 프로젝트는 AI 전문가 양성 과정의 캡스톤 프로젝트입니다.

## 프로젝트 타입
- **타입**: {self.project_type.upper()}
- **생성일**: {self.config['created_at']}
- **버전**: {self.config['version']}

## 디렉토리 구조
```
{self.project_name}/
├── data/           # 데이터 파일
├── models/         # 훈련된 모델
├── src/           # 소스 코드
├── tests/         # 테스트 코드
├── docs/          # 문서
├── notebooks/     # Jupyter 노트북
├── scripts/       # 유틸리티 스크립트
└── config/        # 설정 파일
```

## 설치 및 실행
```bash
# 의존성 설치
pip install -r requirements.txt

# 프로젝트 실행
python src/main.py
```

## 기여 방법
1. 이슈 등록
2. 브랜치 생성
3. 코드 작성
4. 테스트 실행
5. 풀 리퀘스트

## 라이선스
MIT License
"""
        
        readme_file = self.project_path / "README.md"
        with open(readme_file, 'w', encoding='utf-8') as f:
            f.write(readme_content)


class ModelEvaluator:
    """모델 평가 및 성능 측정 클래스"""
    
    def __init__(self):
        self.metrics = {}
        self.results = {}
    
    def evaluate_classification(self, y_true: np.ndarray, y_pred: np.ndarray, 
                              y_prob: Optional[np.ndarray] = None) -> Dict[str, float]:
        """
        분류 모델 평가
        
        Args:
            y_true: 실제 레이블
            y_pred: 예측 레이블
            y_prob: 예측 확률 (선택사항)
            
        Returns:
            평가 메트릭 딕셔너리
        """
        # 빈 데이터 검증
        if len(y_true) == 0 or len(y_pred) == 0:
            raise ValueError("빈 데이터로는 평가를 수행할 수 없습니다.")
        
        from sklearn.metrics import accuracy_score, precision_score, recall_score, f1_score
        
        metrics = {
            'accuracy': accuracy_score(y_true, y_pred),
            'precision': precision_score(y_true, y_pred, average='weighted'),
            'recall': recall_score(y_true, y_pred, average='weighted'),
            'f1_score': f1_score(y_true, y_pred, average='weighted')
        }
        
        self.metrics['classification'] = metrics
        return metrics
    
    def evaluate_regression(self, y_true: np.ndarray, y_pred: np.ndarray) -> Dict[str, float]:
        """
        회귀 모델 평가
        
        Args:
            y_true: 실제 값
            y_pred: 예측 값
            
        Returns:
            평가 메트릭 딕셔너리
        """
        # 빈 데이터 검증
        if len(y_true) == 0 or len(y_pred) == 0:
            raise ValueError("빈 데이터로는 평가를 수행할 수 없습니다.")
        
        from sklearn.metrics import mean_squared_error, mean_absolute_error, r2_score
        
        metrics = {
            'mse': mean_squared_error(y_true, y_pred),
            'rmse': np.sqrt(mean_squared_error(y_true, y_pred)),
            'mae': mean_absolute_error(y_true, y_pred),
            'r2_score': r2_score(y_true, y_pred)
        }
        
        self.metrics['regression'] = metrics
        return metrics
    
    def generate_report(self, output_path: str = "evaluation_report.html") -> str:
        """평가 결과 리포트 생성"""
        try:
            # plotly import 시도
            try:
                import plotly.graph_objects as go
                from plotly.subplots import make_subplots
                plotly_available = True
            except ImportError:
                plotly_available = False
            
            # 리포트 생성 로직
            report_content = f"""
            <html>
            <head>
                <title>모델 평가 리포트</title>
                <style>
                    body {{ font-family: Arial, sans-serif; margin: 20px; }}
                    .metric {{ margin: 10px 0; padding: 10px; background-color: #f5f5f5; }}
                </style>
            </head>
            <body>
                <h1>모델 평가 리포트</h1>
                <p>생성일: {datetime.now().strftime('%Y-%m-%d %H:%M:%S')}</p>
                
                <h2>평가 메트릭</h2>
            """
            
            for metric_type, metrics in self.metrics.items():
                report_content += f"<h3>{metric_type.title()}</h3>"
                for metric_name, value in metrics.items():
                    report_content += f'<div class="metric"><strong>{metric_name}:</strong> {value:.4f}</div>'
            
            report_content += """
            </body>
            </html>
            """
            
            with open(output_path, 'w', encoding='utf-8') as f:
                f.write(report_content)
            
            logger.info(f"평가 리포트가 생성되었습니다: {output_path}")
            return output_path
            
        except Exception as e:
            logger.error(f"리포트 생성 중 오류 발생: {e}")
            return ""


class PortfolioGenerator:
    """포트폴리오 생성 클래스"""
    
    def __init__(self, student_name: str):
        self.student_name = student_name
        self.projects = []
        self.skills = []
    
    def add_project(self, project_info: Dict[str, Any]):
        """프로젝트 정보 추가"""
        self.projects.append(project_info)
    
    def add_skill(self, skill: str, level: str = "Intermediate"):
        """기술 스킬 추가"""
        self.skills.append({"skill": skill, "level": level})
    
    def generate_portfolio(self, output_path: str = "portfolio.html") -> str:
        """포트폴리오 HTML 생성"""
        portfolio_content = f"""
        <!DOCTYPE html>
        <html lang="ko">
        <head>
            <meta charset="UTF-8">
            <meta name="viewport" content="width=device-width, initial-scale=1.0">
            <title>{self.student_name} - AI 포트폴리오</title>
            <style>
                body {{
                    font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
                    line-height: 1.6;
                    margin: 0;
                    padding: 0;
                    background-color: #f4f4f4;
                }}
                .container {{
                    max-width: 1200px;
                    margin: 0 auto;
                    padding: 20px;
                }}
                .header {{
                    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
                    color: white;
                    padding: 60px 20px;
                    text-align: center;
                    border-radius: 10px;
                    margin-bottom: 30px;
                }}
                .section {{
                    background: white;
                    padding: 30px;
                    margin-bottom: 30px;
                    border-radius: 10px;
                    box-shadow: 0 2px 10px rgba(0,0,0,0.1);
                }}
                .project-card {{
                    border: 1px solid #ddd;
                    padding: 20px;
                    margin: 15px 0;
                    border-radius: 8px;
                    background-color: #fafafa;
                }}
                .skill-tag {{
                    display: inline-block;
                    background-color: #667eea;
                    color: white;
                    padding: 5px 15px;
                    border-radius: 20px;
                    margin: 5px;
                    font-size: 14px;
                }}
                h1, h2, h3 {{
                    color: #333;
                }}
            </style>
        </head>
        <body>
            <div class="container">
                <div class="header">
                    <h1>{self.student_name}</h1>
                    <p>AI 전문가 양성 과정 수료생</p>
                </div>
                
                <div class="section">
                    <h2>기술 스킬</h2>
                    <div>
        """
        
        for skill_info in self.skills:
            portfolio_content += f'<span class="skill-tag">{skill_info["skill"]} ({skill_info["level"]})</span>'
        
        portfolio_content += """
                    </div>
                </div>
                
                <div class="section">
                    <h2>프로젝트</h2>
        """
        
        for project in self.projects:
            portfolio_content += f"""
                    <div class="project-card">
                        <h3>{project.get('name', '프로젝트')}</h3>
                        <p><strong>설명:</strong> {project.get('description', '')}</p>
                        <p><strong>기술:</strong> {project.get('technologies', '')}</p>
                        <p><strong>결과:</strong> {project.get('results', '')}</p>
                    </div>
            """
        
        portfolio_content += """
                </div>
            </div>
        </body>
        </html>
        """
        
        with open(output_path, 'w', encoding='utf-8') as f:
            f.write(portfolio_content)
        
        logger.info(f"포트폴리오가 생성되었습니다: {output_path}")
        return output_path


def main():
    """메인 실행 함수"""
    print("=== AI 전문가 양성 과정 캡스톤 프로젝트 ===")
    
    # 프로젝트 생성 예제
    project = CapstoneProject("my_ai_project", "ml")
    if project.create_project_structure():
        print("✅ 프로젝트 구조가 성공적으로 생성되었습니다.")
    
    # 모델 평가 예제
    evaluator = ModelEvaluator()
    print("✅ 모델 평가 도구가 준비되었습니다.")
    
    # 포트폴리오 생성 예제
    portfolio = PortfolioGenerator("홍길동")
    portfolio.add_skill("Python", "Advanced")
    portfolio.add_skill("Machine Learning", "Intermediate")
    portfolio.add_skill("Deep Learning", "Intermediate")
    portfolio.add_skill("FastAPI", "Intermediate")
    
    portfolio.add_project({
        "name": "감정 분석 모델",
        "description": "텍스트 기반 감정 분석 모델 개발",
        "technologies": "Python, TensorFlow, FastAPI",
        "results": "정확도 85% 달성"
    })
    
    portfolio.generate_portfolio()
    print("✅ 포트폴리오가 생성되었습니다.")


if __name__ == "__main__":
    main() 