import os

# advanced_rag_chatbot.py 스크립트의 main 함수를 가져옵니다.
# (실제 환경에서는 모듈로 분리하여 import하는 것이 좋습니다)
from autonomous_agent import main as run_advanced_rag_agent
from dotenv import load_dotenv


def setup_langsmith_environment():
    """
    LangSmith 연동을 위한 환경 변수를 설정하고 사용자에게 안내합니다.
    .env 파일에 다음 변수들이 설정되어 있어야 합니다.

    LANGCHAIN_TRACING_V2="true"
    LANGCHAIN_ENDPOINT="https://api.smith.langchain.com"
    LANGCHAIN_API_KEY="YOUR_LANGSMITH_API_KEY"
    LANGCHAIN_PROJECT="YOUR_PROJECT_NAME" (예: "Advanced RAG Monitoring")
    """
    load_dotenv()

    required_vars = [
        "LANGCHAIN_TRACING_V2",
        "LANGCHAIN_ENDPOINT",
        "LANGCHAIN_API_KEY",
        "LANGCHAIN_PROJECT",
        "OPENAI_API_KEY",  # RAG 에이전트 실행에 필요
        "TAVILY_API_KEY",  # RAG 에이전트 실행에 필요
    ]

    missing_vars = [var for var in required_vars if not os.getenv(var)]

    if missing_vars:
        raise ValueError(
            f"다음 환경 변수가 .env 파일에 설정되지 않았습니다: {', '.join(missing_vars)}"
        )

    if os.getenv("LANGCHAIN_TRACING_V2") != "true":
        print(
            "경고: LANGCHAIN_TRACING_V2가 'true'로 설정되지 않았습니다. LangSmith 추적이 활성화되지 않습니다."
        )

    print("LangSmith 환경 변수 설정이 확인되었습니다.")
    print(f"프로젝트 '{os.getenv('LANGCHAIN_PROJECT')}' 에 실행 결과가 기록됩니다.")


def main():
    """
    LangSmith 환경 설정을 확인하고, 고급 RAG 에이전트를 실행하여
    그 과정을 LangSmith으로 추적하는 실습을 진행합니다.
    """
    print("--- LangSmith LLM 앱 추적 및 디버깅 실습 ---")

    try:
        setup_langsmith_environment()
    except ValueError as e:
        print(f"오류: {e}")
        print("\n실습을 진행하기 전에 .env 파일에 필요한 모든 키를 설정해주세요.")
        print(
            "LangSmith 가입 및 API 키 발급은 https://smith.langchain.com/ 를 참고하세요."
        )
        return

    print("\n이제 'autonomous_agent.py'의 자율 에이전트를 실행합니다.")
    print(
        "실행이 완료된 후, LangSmith 프로젝트 페이지에서 전체 실행 과정을 추적할 수 있습니다."
    )
    print("=" * 50)

    try:
        # autonomous_agent.py의 main 함수를 실행
        run_advanced_rag_agent()

        project_name = os.getenv("LANGCHAIN_PROJECT")
        print("\n" + "=" * 50)
        print("에이전트 실행이 완료되었습니다.")
        print("LangSmith 대시보드에서 결과를 확인하세요.")
        print(
            f"프로젝트 URL (예상): https://smith.langchain.com/o/YOUR_ORGANIZATION/projects/p/{project_name}"
        )

    except Exception as e:
        print(f"\n에이전트 실행 중 오류가 발생했습니다: {e}")
        print("API 키가 유효한지, 관련 라이브러리가 모두 설치되었는지 확인해주세요.")


if __name__ == "__main__":
    main()
