"""
Part 10: AI 전문가 경로 실습
AI 전문가가 되기 위한 고급 기술과 실무 프로젝트 예제
"""

import numpy as np
import pandas as pd
from sklearn.ensemble import RandomForestClassifier
from sklearn.model_selection import train_test_split, cross_val_score
from sklearn.metrics import classification_report, confusion_matrix
import matplotlib.pyplot as plt
import seaborn as sns


class ExpertAIAnalyst:
    """AI 전문가 분석가 클래스"""
    
    def __init__(self, name, specialization):
        self.name = name
        self.specialization = specialization
        self.projects = []
        self.skills = []
        
    def add_skill(self, skill):
        """기술 스택 추가"""
        self.skills.append(skill)
        print(f"기술 추가: {skill}")
        
    def add_project(self, project_name, description, technologies):
        """프로젝트 경험 추가"""
        project = {
            'name': project_name,
            'description': description,
            'technologies': technologies,
            'status': 'completed'
        }
        self.projects.append(project)
        print(f"프로젝트 추가: {project_name}")
        
    def get_expertise_summary(self):
        """전문성 요약"""
        return {
            'name': self.name,
            'specialization': self.specialization,
            'total_skills': len(self.skills),
            'total_projects': len(self.projects),
            'skills': self.skills,
            'projects': [p['name'] for p in self.projects]
        }


class AdvancedMLPipeline:
    """고급 머신러닝 파이프라인"""
    
    def __init__(self):
        self.models = {}
        self.feature_importance = {}
        self.performance_metrics = {}
        
    def create_synthetic_data(self, n_samples=1000, n_features=10):
        """합성 데이터 생성"""
        np.random.seed(42)
        X = np.random.randn(n_samples, n_features)
        y = (X[:, 0] + X[:, 1] > 0).astype(int)  # 간단한 분류 문제
        return X, y
        
    def train_model(self, X, y, model_name="random_forest"):
        """모델 훈련"""
        X_train, X_test, y_train, y_test = train_test_split(
            X, y, test_size=0.2, random_state=42
        )
        
        if model_name == "random_forest":
            model = RandomForestClassifier(n_estimators=100, random_state=42)
        else:
            # 기본값으로 RandomForest 사용
            model = RandomForestClassifier(n_estimators=100, random_state=42)
            
        model.fit(X_train, y_train)
        
        # 성능 평가
        train_score = model.score(X_train, y_train)
        test_score = model.score(X_test, y_test)
        cv_scores = cross_val_score(model, X, y, cv=5)
        
        # 특성 중요도
        if hasattr(model, 'feature_importances_'):
            feature_importance = model.feature_importances_
        else:
            feature_importance = None
            
        # 결과 저장
        self.models[model_name] = model
        self.feature_importance[model_name] = feature_importance
        self.performance_metrics[model_name] = {
            'train_score': train_score,
            'test_score': test_score,
            'cv_mean': cv_scores.mean(),
            'cv_std': cv_scores.std()
        }
        
        return model, self.performance_metrics[model_name]
        
    def analyze_performance(self, model_name):
        """성능 분석"""
        if model_name not in self.performance_metrics:
            print(f"모델 {model_name}이 훈련되지 않았습니다.")
            return None
            
        metrics = self.performance_metrics[model_name]
        print(f"\n=== {model_name} 성능 분석 ===")
        print(f"훈련 정확도: {metrics['train_score']:.4f}")
        print(f"테스트 정확도: {metrics['test_score']:.4f}")
        print(f"교차 검증 평균: {metrics['cv_mean']:.4f} (+/- {metrics['cv_std']*2:.4f})")
        
        return metrics
        
    def plot_feature_importance(self, model_name, top_n=10):
        """특성 중요도 시각화"""
        if model_name not in self.feature_importance:
            print(f"모델 {model_name}의 특성 중요도가 없습니다.")
            return
            
        importance = self.feature_importance[model_name]
        if importance is None:
            print("이 모델은 특성 중요도를 제공하지 않습니다.")
            return
            
        # 상위 N개 특성 선택
        indices = np.argsort(importance)[::-1][:top_n]
        
        plt.figure(figsize=(10, 6))
        plt.title(f'{model_name} - 특성 중요도 (상위 {top_n}개)')
        plt.bar(range(top_n), importance[indices])
        plt.xlabel('특성 인덱스')
        plt.ylabel('중요도')
        plt.xticks(range(top_n), [f'Feature_{i}' for i in indices])
        plt.tight_layout()
        plt.show()


class ExpertPathGuide:
    """AI 전문가 경로 가이드"""
    
    def __init__(self):
        self.career_paths = {
            'ML_Engineer': {
                'description': '머신러닝 모델 개발 및 배포',
                'skills': ['Python', 'TensorFlow', 'PyTorch', 'Docker', 'Kubernetes', 'MLOps'],
                'projects': ['추천 시스템', '이미지 분류', '자연어 처리'],
                'salary_range': '8000-15000만원'
            },
            'Data_Scientist': {
                'description': '데이터 분석 및 인사이트 도출',
                'skills': ['Python', 'R', 'SQL', 'Pandas', 'Scikit-learn', '통계학'],
                'projects': ['고객 세분화', '예측 모델링', 'A/B 테스트'],
                'salary_range': '6000-12000만원'
            },
            'AI_Researcher': {
                'description': '최신 AI 기술 연구 및 개발',
                'skills': ['Python', 'PyTorch', '수학', '논문 작성', '연구 방법론'],
                'projects': ['논문 구현', '새로운 모델 개발', '성능 최적화'],
                'salary_range': '7000-13000만원'
            },
            'MLOps_Engineer': {
                'description': '머신러닝 파이프라인 운영 및 자동화',
                'skills': ['Python', 'Docker', 'Kubernetes', 'CI/CD', '모니터링', '클라우드'],
                'projects': ['ML 파이프라인 구축', '모델 배포 자동화', '성능 모니터링'],
                'salary_range': '8000-14000만원'
            }
        }
        
    def get_career_path(self, path_name):
        """특정 경력 경로 정보 조회"""
        if path_name not in self.career_paths:
            print(f"경력 경로 '{path_name}'을 찾을 수 없습니다.")
            return None
            
        return self.career_paths[path_name]
        
    def recommend_path(self, interests, current_skills):
        """관심사와 현재 기술을 바탕으로 경력 경로 추천"""
        scores = {}
        
        for path_name, path_info in self.career_paths.items():
            score = 0
            # 관심사 매칭
            for interest in interests:
                if interest.lower() in path_info['description'].lower():
                    score += 2
                    
            # 기술 매칭
            for skill in current_skills:
                if skill in path_info['skills']:
                    score += 1
                    
            scores[path_name] = score
            
        # 점수 순으로 정렬
        sorted_paths = sorted(scores.items(), key=lambda x: x[1], reverse=True)
        
        print("\n=== 경력 경로 추천 ===")
        for path_name, score in sorted_paths:
            path_info = self.career_paths[path_name]
            print(f"\n{path_name} (점수: {score})")
            print(f"설명: {path_info['description']}")
            print(f"필요 기술: {', '.join(path_info['skills'])}")
            print(f"연봉 범위: {path_info['salary_range']}")
            
        return sorted_paths


def main():
    """메인 실행 함수"""
    print("=== AI 전문가 경로 실습 ===\n")
    
    # 1. 전문가 분석가 생성
    analyst = ExpertAIAnalyst("김AI", "머신러닝 엔지니어")
    
    # 기술 스택 추가
    analyst.add_skill("Python")
    analyst.add_skill("TensorFlow")
    analyst.add_skill("Docker")
    analyst.add_skill("Kubernetes")
    analyst.add_skill("MLOps")
    
    # 프로젝트 경험 추가
    analyst.add_project(
        "추천 시스템 개발",
        "사용자 행동 데이터를 기반으로 한 개인화 추천 시스템",
        ["Python", "TensorFlow", "Docker", "Redis"]
    )
    
    analyst.add_project(
        "이미지 분류 모델",
        "CNN을 활용한 의료 이미지 분류 시스템",
        ["Python", "PyTorch", "OpenCV", "Docker"]
    )
    
    # 전문성 요약
    summary = analyst.get_expertise_summary()
    print(f"\n=== {summary['name']} 전문성 요약 ===")
    print(f"전문 분야: {summary['specialization']}")
    print(f"보유 기술: {summary['total_skills']}개")
    print(f"프로젝트 경험: {summary['total_projects']}개")
    
    # 2. 고급 ML 파이프라인 실행
    print("\n=== 고급 ML 파이프라인 실행 ===")
    pipeline = AdvancedMLPipeline()
    
    # 합성 데이터 생성
    X, y = pipeline.create_synthetic_data(n_samples=1000, n_features=10)
    print(f"데이터 생성 완료: {X.shape[0]}개 샘플, {X.shape[1]}개 특성")
    
    # 모델 훈련
    model, metrics = pipeline.train_model(X, y, "random_forest")
    print("모델 훈련 완료")
    
    # 성능 분석
    pipeline.analyze_performance("random_forest")
    
    # 3. 경력 경로 가이드
    print("\n=== AI 전문가 경력 경로 가이드 ===")
    guide = ExpertPathGuide()
    
    # 현재 관심사와 기술
    interests = ["머신러닝", "자동화", "배포"]
    current_skills = ["Python", "Docker", "기본 통계"]
    
    # 경력 경로 추천
    guide.recommend_path(interests, current_skills)
    
    # 4. 학습 로드맵 제시
    print("\n=== AI 전문가 학습 로드맵 ===")
    roadmap = {
        "1단계 (기초)": [
            "Python 프로그래밍 마스터",
            "수학 기초 (선형대수, 미적분, 통계)",
            "머신러닝 기초 알고리즘 이해"
        ],
        "2단계 (중급)": [
            "딥러닝 프레임워크 학습 (TensorFlow/PyTorch)",
            "실제 프로젝트 수행",
            "데이터 전처리 및 특성 엔지니어링"
        ],
        "3단계 (고급)": [
            "MLOps 및 모델 배포",
            "클라우드 플랫폼 활용",
            "연구 논문 읽기 및 구현"
        ],
        "4단계 (전문가)": [
            "새로운 알고리즘 개발",
            "팀 리딩 및 멘토링",
            "컨퍼런스 발표 및 논문 작성"
        ]
    }
    
    for stage, tasks in roadmap.items():
        print(f"\n{stage}:")
        for task in tasks:
            print(f"  - {task}")
    
    print("\n=== 실습 완료 ===")
    print("AI 전문가가 되기 위한 여정을 시작하세요!")


if __name__ == "__main__":
    main() 