#!/bin/bash

# AI 전문가 양성 과정 - 환경 설정 자동화 스크립트 (Linux/macOS)
# 이 스크립트는 필요한 Python 패키지들을 자동으로 설치합니다.

set -e  # 오류 발생 시 스크립트 중단

echo "🚀 AI 전문가 양성 과정 환경 설정을 시작합니다..."
echo "=========================================="

# 색상 코드 정의
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
NC='\033[0m' # No Color

# Python 버전 확인
check_python() {
    echo -e "${BLUE}Python 버전을 확인합니다...${NC}"
    
    if command -v python3 &> /dev/null; then
        PYTHON_VERSION=$(python3 --version | cut -d ' ' -f 2)
        echo -e "${GREEN}Python 버전: $PYTHON_VERSION${NC}"
        
        # Python 3.8 이상 필요
        if python3 -c "import sys; exit(0 if sys.version_info >= (3, 8) else 1)"; then
            echo -e "${GREEN}✓ Python 버전이 요구사항을 만족합니다.${NC}"
        else
            echo -e "${RED}✗ Python 3.8 이상이 필요합니다. 현재 버전: $PYTHON_VERSION${NC}"
            exit 1
        fi
    else
        echo -e "${RED}✗ Python3가 설치되어 있지 않습니다.${NC}"
        echo "Python3를 먼저 설치해주세요."
        exit 1
    fi
}

# pip 업그레이드
upgrade_pip() {
    echo -e "${BLUE}pip를 최신 버전으로 업그레이드합니다...${NC}"
    python3 -m pip install --upgrade pip
    echo -e "${GREEN}✓ pip 업그레이드 완료${NC}"
}

# 가상환경 생성 및 활성화
setup_venv() {
    echo -e "${BLUE}가상환경을 설정합니다...${NC}"
    
    if [ ! -d "venv" ]; then
        echo "가상환경을 생성합니다..."
        python3 -m venv venv
        echo -e "${GREEN}✓ 가상환경 생성 완료${NC}"
    else
        echo -e "${YELLOW}가상환경이 이미 존재합니다.${NC}"
    fi
    
    echo "가상환경을 활성화합니다..."
    source venv/bin/activate
    echo -e "${GREEN}✓ 가상환경 활성화 완료${NC}"
}

# 기본 패키지 설치
install_basic_packages() {
    echo -e "${BLUE}기본 패키지들을 설치합니다...${NC}"
    
    pip install --upgrade pip setuptools wheel
    
    # requirements.txt가 있다면 사용, 없다면 기본 패키지 설치
    if [ -f "requirements.txt" ]; then
        echo "requirements.txt에서 패키지를 설치합니다..."
        pip install -r requirements.txt
    else
        echo "기본 패키지들을 설치합니다..."
        pip install \
            "numpy>=1.21.0" \
            "pandas>=1.3.0" \
            "matplotlib>=3.4.0" \
            "seaborn>=0.11.0" \
            "scikit-learn>=1.0.0" \
            "jupyter>=1.0.0" \
            "notebook>=6.4.0" \
            "ipykernel>=6.0.0"
    fi
    
    echo -e "${GREEN}✓ 기본 패키지 설치 완료${NC}"
}

# Jupyter 커널 등록
setup_jupyter_kernel() {
    echo -e "${BLUE}Jupyter 커널을 등록합니다...${NC}"
    
    python3 -m ipykernel install --user --name=ai_course --display-name="AI Course"
    
    echo -e "${GREEN}✓ Jupyter 커널 등록 완료${NC}"
}

# 설치 완료 확인
verify_installation() {
    echo -e "${BLUE}설치된 패키지를 확인합니다...${NC}"
    
    python3 -c "
import sys
print(f'Python 버전: {sys.version}')

packages = ['numpy', 'pandas', 'matplotlib', 'seaborn', 'sklearn', 'jupyter']

for package in packages:
    try:
        __import__(package)
        print(f'✓ {package}: 설치됨')
    except ImportError:
        print(f'✗ {package}: 설치되지 않음')
"
    
    echo -e "${GREEN}✓ 패키지 확인 완료${NC}"
}

# 사용법 안내
show_usage() {
    echo -e "${BLUE}=========================================="
    echo -e "🎉 환경 설정이 완료되었습니다!"
    echo -e "==========================================${NC}"
    echo ""
    echo -e "${YELLOW}다음 명령어로 가상환경을 활성화하세요:${NC}"
    echo "source venv/bin/activate"
    echo ""
    echo -e "${YELLOW}Jupyter Notebook을 시작하려면:${NC}"
    echo "jupyter notebook"
    echo ""
    echo -e "${YELLOW}코드를 실행하려면:${NC}"
    echo "cd source_code/02_python_core_syntax"
    echo "python part_2_python_core_syntax.py"
}

# 메인 실행 함수
main() {
    echo -e "${GREEN}AI 전문가 양성 과정 환경 설정${NC}"
    echo ""
    
    check_python
    upgrade_pip
    setup_venv
    install_basic_packages
    setup_jupyter_kernel
    verify_installation
    show_usage
    
    echo -e "${GREEN}🎉 모든 설정이 완료되었습니다!${NC}"
}

# 스크립트 실행
main "$@" 