#!/bin/bash

# AI 전문가 양성 과정 - 테스트 실행 자동화 스크립트
# 이 스크립트는 프로젝트의 모든 테스트를 자동으로 실행합니다.

set -e  # 오류 발생 시 스크립트 중단

echo "🧪 AI 전문가 양성 과정 테스트를 시작합니다..."
echo "=========================================="

# 색상 코드 정의
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
NC='\033[0m' # No Color

# 테스트 결과 변수
TOTAL_TESTS=0
PASSED_TESTS=0
FAILED_TESTS=0

# Python 및 pytest 설치 확인
check_requirements() {
    echo -e "${BLUE}테스트 환경을 확인합니다...${NC}"
    
    if ! command -v python3 &> /dev/null; then
        echo -e "${RED}✗ Python3가 설치되어 있지 않습니다.${NC}"
        exit 1
    fi
    
    # pytest가 설치되어 있는지 확인
    if ! python3 -c "import pytest" 2>/dev/null; then
        echo -e "${YELLOW}pytest가 설치되어 있지 않습니다. 설치합니다...${NC}"
        pip install pytest pytest-cov
    fi
    
    echo -e "${GREEN}✓ 테스트 환경 확인 완료${NC}"
}

# 특정 디렉토리의 테스트 실행
run_tests_in_directory() {
    local test_dir=$1
    local module_name=$2
    
    if [ -d "$test_dir" ] && [ -n "$(find $test_dir -name 'test_*.py' -type f)" ]; then
        echo -e "${BLUE}📦 $module_name 테스트를 실행합니다...${NC}"
        echo "테스트 경로: $test_dir"
        
        if python3 -m pytest "$test_dir" -v --tb=short; then
            echo -e "${GREEN}✓ $module_name 테스트 통과${NC}"
            PASSED_TESTS=$((PASSED_TESTS + 1))
        else
            echo -e "${RED}✗ $module_name 테스트 실패${NC}"
            FAILED_TESTS=$((FAILED_TESTS + 1))
        fi
        
        TOTAL_TESTS=$((TOTAL_TESTS + 1))
        echo ""
    else
        echo -e "${YELLOW}⚠ $module_name 테스트 파일이 없습니다. ($test_dir)${NC}"
    fi
}

# 코드 품질 검사 (선택적)
run_code_quality_checks() {
    echo -e "${BLUE}📋 코드 품질을 검사합니다...${NC}"
    
    # flake8이 설치되어 있으면 실행
    if python3 -c "import flake8" 2>/dev/null; then
        echo "flake8으로 코드 스타일을 검사합니다..."
        if flake8 source_code/ --max-line-length=88 --extend-ignore=E203,W503; then
            echo -e "${GREEN}✓ 코드 스타일 검사 통과${NC}"
        else
            echo -e "${YELLOW}⚠ 코드 스타일 개선이 필요합니다${NC}"
        fi
    fi
    
    # black이 설치되어 있으면 체크 모드로 실행
    if python3 -c "import black" 2>/dev/null; then
        echo "black으로 코드 포맷팅을 검사합니다..."
        if black --check source_code/ 2>/dev/null; then
            echo -e "${GREEN}✓ 코드 포맷팅 검사 통과${NC}"
        else
            echo -e "${YELLOW}⚠ 코드 포맷팅 개선이 필요합니다${NC}"
        fi
    fi
    
    echo ""
}

# 커버리지 리포트 생성
generate_coverage_report() {
    echo -e "${BLUE}📊 테스트 커버리지를 확인합니다...${NC}"
    
    if python3 -c "import coverage" 2>/dev/null; then
        echo "커버리지 리포트를 생성합니다..."
        python3 -m pytest source_code/ --cov=source_code --cov-report=term-missing --cov-report=html 2>/dev/null || true
        
        if [ -d "htmlcov" ]; then
            echo -e "${GREEN}✓ 커버리지 리포트가 htmlcov/ 폴더에 생성되었습니다${NC}"
        fi
    else
        echo -e "${YELLOW}⚠ coverage 패키지가 설치되지 않아 커버리지 리포트를 생략합니다${NC}"
    fi
    
    echo ""
}

# 테스트 결과 요약
show_test_summary() {
    echo -e "${BLUE}=========================================="
    echo -e "🧪 테스트 결과 요약"
    echo -e "==========================================${NC}"
    echo -e "전체 테스트 모듈: ${TOTAL_TESTS}개"
    echo -e "${GREEN}✓ 통과: ${PASSED_TESTS}개${NC}"
    
    if [ $FAILED_TESTS -gt 0 ]; then
        echo -e "${RED}✗ 실패: ${FAILED_TESTS}개${NC}"
        echo ""
        echo -e "${RED}⚠ 일부 테스트가 실패했습니다. 위의 로그를 확인해주세요.${NC}"
        return 1
    else
        echo -e "${GREEN}🎉 모든 테스트가 통과했습니다!${NC}"
        return 0
    fi
}

# 메인 실행 함수
main() {
    echo -e "${GREEN}AI 전문가 양성 과정 테스트 실행${NC}"
    echo ""
    
    # 현재 디렉토리 확인
    if [ ! -d "source_code" ]; then
        echo -e "${RED}✗ source_code 디렉토리가 없습니다. 프로젝트 루트에서 실행해주세요.${NC}"
        exit 1
    fi
    
    check_requirements
    
    # 각 모듈별 테스트 실행
    run_tests_in_directory "source_code/05_ai_core_libraries/tests" "AI 핵심 라이브러리"
    run_tests_in_directory "source_code/06_machine_learning/tests" "머신러닝"
    run_tests_in_directory "source_code/07_deep_learning/tests" "딥러닝"
    run_tests_in_directory "source_code/08_model_serving_with_fastapi/tests" "FastAPI 모델 서빙"
    run_tests_in_directory "source_code/09_production_ready_api/app/tests" "프로덕션 API"
    
    # 추가 품질 검사 (선택적)
    if [ "$1" = "--quality" ]; then
        run_code_quality_checks
    fi
    
    # 커버리지 리포트 (선택적)
    if [ "$1" = "--coverage" ] || [ "$2" = "--coverage" ]; then
        generate_coverage_report
    fi
    
    # 결과 요약
    if show_test_summary; then
        echo -e "${GREEN}✅ 모든 테스트가 성공적으로 완료되었습니다!${NC}"
        exit 0
    else
        echo -e "${RED}❌ 일부 테스트가 실패했습니다.${NC}"
        exit 1
    fi
}

# 사용법 표시
show_usage() {
    echo "사용법: $0 [옵션]"
    echo ""
    echo "옵션:"
    echo "  --quality    코드 품질 검사도 함께 실행"
    echo "  --coverage   테스트 커버리지 리포트 생성"
    echo ""
    echo "예시:"
    echo "  $0                    # 기본 테스트만 실행"
    echo "  $0 --quality          # 테스트 + 코드 품질 검사"
    echo "  $0 --coverage         # 테스트 + 커버리지 리포트"
    echo "  $0 --quality --coverage # 모든 검사 실행"
}

# 인자 처리
if [ "$1" = "--help" ] || [ "$1" = "-h" ]; then
    show_usage
    exit 0
fi

# 스크립트 실행
main "$@" 