#!/bin/bash

# AI 전문가 양성 과정 - 코드 포맷팅 스크립트
# 이 스크립트는 프로젝트의 Python 코드를 자동으로 포맷팅합니다.

set -e  # 오류 발생 시 스크립트 중단

echo "🎨 코드 포맷팅을 시작합니다..."
echo "=========================================="

# 색상 코드 정의
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
NC='\033[0m' # No Color

# 필요한 도구들이 설치되어 있는지 확인
check_tools() {
    echo -e "${BLUE}필요한 도구들을 확인합니다...${NC}"
    
    TOOLS_MISSING=false
    
    if ! command -v black &> /dev/null; then
        echo -e "${RED}✗ black이 설치되어 있지 않습니다.${NC}"
        TOOLS_MISSING=true
    else
        echo -e "${GREEN}✓ black 설치됨${NC}"
    fi
    
    if ! command -v isort &> /dev/null; then
        echo -e "${RED}✗ isort가 설치되어 있지 않습니다.${NC}"
        TOOLS_MISSING=true
    else
        echo -e "${GREEN}✓ isort 설치됨${NC}"
    fi
    
    if ! command -v flake8 &> /dev/null; then
        echo -e "${YELLOW}⚠ flake8이 설치되어 있지 않습니다. (선택사항)${NC}"
    else
        echo -e "${GREEN}✓ flake8 설치됨${NC}"
    fi
    
    if [ "$TOOLS_MISSING" = true ]; then
        echo -e "${YELLOW}필요한 도구를 설치하시겠습니까? (y/N)${NC}"
        read -r response
        
        if [[ "$response" =~ ^[Yy]$ ]]; then
            echo -e "${BLUE}필요한 도구들을 설치합니다...${NC}"
            pip install black isort flake8
            echo -e "${GREEN}✓ 도구 설치 완료${NC}"
        else
            echo -e "${RED}필요한 도구가 설치되지 않아 스크립트를 종료합니다.${NC}"
            exit 1
        fi
    fi
}

# import 문 정렬 (isort)
sort_imports() {
    echo -e "${BLUE}import 문을 정렬합니다...${NC}"
    
    # source_code 디렉토리의 Python 파일들 처리
    if [ -d "source_code" ]; then
        find source_code -name "*.py" -exec isort {} \;
        echo -e "${GREEN}✓ source_code 디렉토리 처리 완료${NC}"
    fi
    
    # datasets/utils 디렉토리 처리
    if [ -d "datasets/utils" ]; then
        find datasets/utils -name "*.py" -exec isort {} \;
        echo -e "${GREEN}✓ datasets/utils 디렉토리 처리 완료${NC}"
    fi
    
    # 루트 디렉토리의 Python 파일들 처리
    find . -maxdepth 1 -name "*.py" -exec isort {} \;
    
    echo -e "${GREEN}✓ import 문 정렬 완료${NC}"
}

# 코드 포맷팅 (black)
format_code() {
    echo -e "${BLUE}코드를 포맷팅합니다...${NC}"
    
    # source_code 디렉토리의 Python 파일들 처리
    if [ -d "source_code" ]; then
        black source_code/ --line-length=88
        echo -e "${GREEN}✓ source_code 디렉토리 처리 완료${NC}"
    fi
    
    # datasets/utils 디렉토리 처리
    if [ -d "datasets/utils" ]; then
        black datasets/utils/ --line-length=88
        echo -e "${GREEN}✓ datasets/utils 디렉토리 처리 완료${NC}"
    fi
    
    # 루트 디렉토리의 Python 파일들 처리
    black *.py --line-length=88 2>/dev/null || true
    
    echo -e "${GREEN}✓ 코드 포맷팅 완료${NC}"
}

# 코드 스타일 검사 (flake8)
check_style() {
    if command -v flake8 &> /dev/null; then
        echo -e "${BLUE}코드 스타일을 검사합니다...${NC}"
        
        # flake8 설정
        FLAKE8_CONFIG="--max-line-length=88 --extend-ignore=E203,W503"
        
        # source_code 디렉토리 검사
        if [ -d "source_code" ]; then
            echo -e "${YELLOW}source_code 디렉토리 검사 중...${NC}"
            flake8 source_code/ $FLAKE8_CONFIG || echo -e "${YELLOW}⚠ source_code에서 스타일 이슈 발견${NC}"
        fi
        
        # datasets/utils 디렉토리 검사
        if [ -d "datasets/utils" ]; then
            echo -e "${YELLOW}datasets/utils 디렉토리 검사 중...${NC}"
            flake8 datasets/utils/ $FLAKE8_CONFIG || echo -e "${YELLOW}⚠ datasets/utils에서 스타일 이슈 발견${NC}"
        fi
        
        echo -e "${GREEN}✓ 코드 스타일 검사 완료${NC}"
    else
        echo -e "${YELLOW}flake8이 설치되어 있지 않아 스타일 검사를 건너뜁니다.${NC}"
    fi
}

# 파일 권한 설정
set_permissions() {
    echo -e "${BLUE}스크립트 파일 권한을 설정합니다...${NC}"
    
    # scripts 디렉토리의 스크립트 파일들에 실행 권한 부여
    if [ -d "scripts" ]; then
        find scripts -name "*.sh" -exec chmod +x {} \;
        echo -e "${GREEN}✓ 스크립트 파일 권한 설정 완료${NC}"
    fi
}

# 포맷팅 결과 보고
show_summary() {
    echo -e "${BLUE}=========================================="
    echo -e "🎉 코드 포맷팅이 완료되었습니다!"
    echo -e "==========================================${NC}"
    echo ""
    echo -e "${GREEN}수행된 작업:${NC}"
    echo "✓ import 문 정렬 (isort)"
    echo "✓ 코드 포맷팅 (black)"
    if command -v flake8 &> /dev/null; then
        echo "✓ 코드 스타일 검사 (flake8)"
    fi
    echo "✓ 스크립트 파일 권한 설정"
    echo ""
    echo -e "${YELLOW}다음 파일들이 처리되었습니다:${NC}"
    echo "- source_code/ 디렉토리의 모든 .py 파일"
    echo "- datasets/utils/ 디렉토리의 모든 .py 파일"
    echo "- 루트 디렉토리의 .py 파일들"
    echo ""
    echo -e "${BLUE}Git으로 변경사항을 확인해보세요:${NC}"
    echo "git diff"
}

# 메인 실행 함수
main() {
    echo -e "${GREEN}AI 전문가 양성 과정 - 코드 포맷팅${NC}"
    echo ""
    
    check_tools
    sort_imports
    format_code
    check_style
    set_permissions
    show_summary
    
    echo -e "${GREEN}🎨 모든 포맷팅 작업이 완료되었습니다!${NC}"
}

# 도움말 표시
show_help() {
    echo "사용법: $0 [옵션]"
    echo ""
    echo "옵션:"
    echo "  -h, --help     이 도움말을 표시합니다"
    echo "  --check        포맷팅 없이 검사만 수행합니다"
    echo "  --imports-only import 문 정렬만 수행합니다"
    echo "  --format-only  코드 포맷팅만 수행합니다"
    echo ""
    echo "예제:"
    echo "  $0                # 전체 포맷팅 수행"
    echo "  $0 --check        # 검사만 수행"
    echo "  $0 --imports-only # import 문 정렬만 수행"
}

# 명령행 인수 처리
case "${1:-}" in
    -h|--help)
        show_help
        exit 0
        ;;
    --check)
        echo -e "${BLUE}검사 모드로 실행합니다...${NC}"
        check_tools
        check_style
        ;;
    --imports-only)
        echo -e "${BLUE}import 문 정렬만 수행합니다...${NC}"
        check_tools
        sort_imports
        ;;
    --format-only)
        echo -e "${BLUE}코드 포맷팅만 수행합니다...${NC}"
        check_tools
        format_code
        ;;
    "")
        main
        ;;
    *)
        echo -e "${RED}알 수 없는 옵션: $1${NC}"
        show_help
        exit 1
        ;;
esac 