

# AI 전문가 양성 과정 - 트러블슈팅 가이드

## 📋 개요
AI 전문가 양성 과정에서 자주 발생하는 문제들과 해결 방법을 정리한 문서입니다.

---

## 🔧 환경 설정 문제

### Python 버전 관련 문제

#### 문제: Python 버전이 맞지 않음
```bash
# 오류 메시지
Python 3.7 이하 버전에서 발생하는 문제들
```

**해결 방법:**
```bash
# Python 3.8 이상 설치 확인
python --version

# pyenv를 사용한 Python 버전 관리 (Linux/macOS)
curl https://pyenv.run | bash
pyenv install 3.9.16
pyenv global 3.9.16

# Windows에서 Python 설치
# https://www.python.org/downloads/ 에서 다운로드
```

#### 문제: pip 설치 오류
```bash
# 오류 메시지
pip: command not found
```

**해결 방법:**
```bash
# pip 설치 확인
python -m ensurepip --upgrade

# pip 업그레이드
python -m pip install --upgrade pip
```

### 가상환경 관련 문제

#### 문제: 가상환경 활성화 안됨
**해결 방법:**
```bash
# Linux/macOS
source venv/bin/activate

# Windows
venv\Scripts\activate

# 또는 PowerShell에서
venv\Scripts\Activate.ps1
```

#### 문제: 가상환경에서 패키지 설치 오류
**해결 방법:**
```bash
# 가상환경이 활성화되었는지 확인
which python  # Linux/macOS
where python  # Windows

# 패키지 설치 시 --user 옵션 제거
pip install package_name  # --user 없이
```

---

## 📦 패키지 설치 문제

### 일반적인 설치 오류

#### 문제: 패키지 의존성 충돌
```bash
# 오류 메시지
ERROR: pip's dependency resolver does not currently take into account all the packages that are installed.
```

**해결 방법:**
```bash
# 가상환경 재생성
rm -rf venv
python -m venv venv
source venv/bin/activate  # Linux/macOS
pip install -r requirements.txt
```

#### 문제: 컴파일 오류 (C++ 확장)
```bash
# 오류 메시지
Microsoft Visual C++ 14.0 is required
```

**해결 방법:**
```bash
# Windows: Build Tools 설치
# https://visualstudio.microsoft.com/visual-cpp-build-tools/

# Linux: 개발 도구 설치
sudo apt-get install build-essential python3-dev

# macOS: Xcode Command Line Tools 설치
xcode-select --install
```

### 딥러닝 프레임워크 문제

#### 문제: PyTorch CUDA 버전 미매치
```python
# 오류 메시지
RuntimeError: CUDA error: no kernel image is available for execution on the device
```

**해결 방법:**
```bash
# CUDA 버전 확인
nvidia-smi

# PyTorch 버전에 맞는 CUDA 버전 설치
# https://pytorch.org/get-started/locally/
pip install torch torchvision torchaudio --index-url https://download.pytorch.org/whl/cu118
```

#### 문제: TensorFlow GPU 인식 안됨
```python
# 확인 코드
import tensorflow as tf
print(tf.config.list_physical_devices('GPU'))  # 빈 리스트 반환
```

**해결 방법:**
```bash
# CUDA 및 cuDNN 설치 확인
nvidia-smi
cat /usr/local/cuda/version.txt

# TensorFlow GPU 버전 재설치
pip uninstall tensorflow
pip install tensorflow[and-cuda]
```

---

## 🐳 Docker 관련 문제

### Docker 설치 및 실행 문제

#### 문제: Docker 명령어 권한 오류 (Linux)
```bash
# 오류 메시지
Got permission denied while trying to connect to the Docker daemon socket
```

**해결 방법:**
```bash
# 사용자를 docker 그룹에 추가
sudo usermod -aG docker $USER
newgrp docker

# 또는 sudo 사용
sudo docker-compose up
```

#### 문제: Windows에서 Docker Desktop 설치 오류
**해결 방법:**
1. WSL2 활성화
2. Hyper-V 기능 활성화
3. BIOS에서 가상화 기능 활성화

### Docker Compose 문제

#### 문제: 포트 충돌
```bash
# 오류 메시지
ERROR: for ai-course-dev  Cannot start service ai-course-dev: Ports are not available
```

**해결 방법:**
```bash
# 사용 중인 포트 확인
netstat -tulpn | grep :8888

# 프로세스 종료
sudo kill -9 <PID>

# 또는 docker-compose.yml에서 포트 변경
ports:
  - "8889:8888"  # 호스트 포트 변경
```

#### 문제: 볼륨 마운트 문제
**해결 방법:**
```bash
# 권한 문제 해결
sudo chown -R $USER:$USER ./

# Windows에서 드라이브 공유 설정 확인
# Docker Desktop -> Settings -> Resources -> File Sharing
```

### GPU Docker 문제

#### 문제: nvidia-docker 설치 안됨
**해결 방법:**
```bash
# NVIDIA Docker 런타임 설치
distribution=$(. /etc/os-release;echo $ID$VERSION_ID)
curl -s -L https://nvidia.github.io/nvidia-docker/gpgkey | sudo apt-key add -
curl -s -L https://nvidia.github.io/nvidia-docker/$distribution/nvidia-docker.list | sudo tee /etc/apt/sources.list.d/nvidia-docker.list

sudo apt-get update && sudo apt-get install -y nvidia-docker2
sudo systemctl restart docker
```

---

## 💻 Jupyter 관련 문제

### Jupyter Notebook/Lab 실행 문제

#### 문제: Jupyter 서버 시작 안됨
```bash
# 오류 메시지
jupyter: command not found
```

**해결 방법:**
```bash
# Jupyter 설치 확인
pip install jupyter jupyterlab

# PATH 확인
echo $PATH

# 가상환경에서 설치했다면 가상환경 활성화 확인
```

#### 문제: 브라우저에서 접속 안됨
**해결 방법:**
```bash
# 토큰 확인
jupyter lab list

# 수동으로 서버 시작 (디버깅 모드)
jupyter lab --ip=0.0.0.0 --port=8888 --no-browser --debug

# 방화벽 설정 확인 (포트 8888 열기)
```

### 커널 관련 문제

#### 문제: Python 커널 인식 안됨
**해결 방법:**
```bash
# 커널 등록
python -m ipykernel install --user --name=ai_course --display-name="AI Course"

# 설치된 커널 확인
jupyter kernelspec list

# 문제가 있는 커널 제거
jupyter kernelspec uninstall ai_course
```

#### 문제: 패키지 import 오류 (Jupyter에서)
```python
# 오류 메시지
ModuleNotFoundError: No module named 'package_name'
```

**해결 방법:**
```python
# Jupyter에서 시스템 체크
import sys
print(sys.executable)
print(sys.path)

# 패키지 설치 확인
!pip list | grep package_name

# 필요시 Jupyter에서 직접 설치
!pip install package_name
```

---

## 🤖 머신러닝/딥러닝 실행 문제

### 메모리 관련 문제

#### 문제: OutOfMemoryError (GPU)
```python
# 오류 메시지
RuntimeError: CUDA out of memory
```

**해결 방법:**
```python
# PyTorch - 배치 크기 줄이기
batch_size = 16  # 32에서 16으로 감소

# PyTorch - GPU 메모리 정리
import torch
torch.cuda.empty_cache()

# TensorFlow - GPU 메모리 증가 설정
import tensorflow as tf
gpus = tf.config.experimental.list_physical_devices('GPU')
if gpus:
    try:
        tf.config.experimental.set_memory_growth(gpus[0], True)
    except RuntimeError as e:
        print(e)
```

#### 문제: 시스템 메모리 부족
**해결 방법:**
```python
# 데이터 청크 단위로 처리
import pandas as pd
chunksize = 10000
for chunk in pd.read_csv('large_file.csv', chunksize=chunksize):
    # 처리 로직
    pass

# Dask 사용 (대용량 데이터)
import dask.dataframe as dd
df = dd.read_csv('large_file.csv')
```

### 모델 학습 문제

#### 문제: 학습이 매우 느림
**해결 방법:**
```python
# 데이터 로딩 최적화
from torch.utils.data import DataLoader
dataloader = DataLoader(dataset, batch_size=32, num_workers=4, pin_memory=True)

# Mixed Precision Training (PyTorch)
from torch.cuda.amp import autocast, GradScaler
scaler = GradScaler()

# 프로파일링으로 병목 지점 찾기
import cProfile
cProfile.run('your_training_function()')
```

---

## 🌐 네트워크 관련 문제

### 데이터 다운로드 문제

#### 문제: 모델/데이터셋 다운로드 실패
```python
# 오류 메시지
urllib.error.URLError: <urlopen error [Errno 11001] getaddrinfo failed>
```

**해결 방법:**
```python
# 프록시 설정
import os
os.environ['HTTP_PROXY'] = 'http://proxy.company.com:8080'
os.environ['HTTPS_PROXY'] = 'https://proxy.company.com:8080'

# Hugging Face 캐시 디렉토리 설정
os.environ['TRANSFORMERS_CACHE'] = '/path/to/cache'

# 수동 다운로드 후 로컬 로드
from transformers import AutoModel
model = AutoModel.from_pretrained('/local/path/to/model')
```

### API 관련 문제

#### 문제: OpenAI API 키 오류
```python
# 오류 메시지
openai.error.AuthenticationError: No API key provided
```

**해결 방법:**
```python
# 환경변수 설정
import os
os.environ['OPENAI_API_KEY'] = 'your-api-key-here'

# 또는 .env 파일 사용
from dotenv import load_dotenv
load_dotenv()

# API 키 확인
import openai
openai.api_key = os.getenv('OPENAI_API_KEY')
```

---

## 🔍 일반적인 디버깅 방법

### 로그 확인
```python
# Python 로깅 설정
import logging
logging.basicConfig(level=logging.DEBUG)

# 상세 오류 정보
import traceback
try:
    # 문제가 되는 코드
    pass
except Exception as e:
    traceback.print_exc()
```

### 환경 정보 수집
```python
# 시스템 정보 수집 스크립트
import sys
import platform
import pkg_resources

print(f"Python 버전: {sys.version}")
print(f"플랫폼: {platform.platform()}")
print(f"아키텍처: {platform.architecture()}")

# 설치된 패키지 버전
packages = ['torch', 'tensorflow', 'numpy', 'pandas']
for package in packages:
    try:
        version = pkg_resources.get_distribution(package).version
        print(f"{package}: {version}")
    except:
        print(f"{package}: 설치되지 않음")
```

---

## 📞 추가 도움이 필요한 경우

### 커뮤니티 지원
- **GitHub Issues**: 프로젝트 저장소에 이슈 등록
- **Stack Overflow**: 기술적 질문
- **Discord/Slack**: 실시간 채팅 지원

### 문의하기 전 체크리스트
1. [ ] 오류 메시지 전체 복사
2. [ ] 운영체제 및 Python 버전 확인
3. [ ] 가상환경 활성화 상태 확인
4. [ ] 설치된 패키지 버전 확인
5. [ ] 재현 가능한 최소 예제 작성

---

**마지막 업데이트:** 2024-12-28  
**담당자:** AI Expert Course Team 