// 전역 변수
let temperatureChart = null;
const API_BASE_URL = 'http://192.168.0.71:8004';

// 페이지 로드 시 초기화
document.addEventListener('DOMContentLoaded', function() {
    updateDashboard();
    // 5초마다 대시보드 업데이트
    setInterval(updateDashboard, 5000);
});

// 대시보드 업데이트
async function updateDashboard() {
    await Promise.all([
        updateGPUStatus(),
        updateCoolingStatus(),
        updateTemperatureChart(),
        updateAlerts()
    ]);
}

// GPU 상태 업데이트
async function updateGPUStatus() {
    try {
        const response = await fetch(`${API_BASE_URL}/api/gpu/status`);
        if (!response.ok) {
            throw new Error(`HTTP error! status: ${response.status}`);
        }
        const gpuInfo = await response.json();
        
        const gpuStatusDiv = document.getElementById('gpuStatus');
        gpuStatusDiv.innerHTML = '';
        
        if (!Array.isArray(gpuInfo)) {
            throw new Error('GPU 정보가 배열 형태가 아닙니다.');
        }
        
        gpuInfo.forEach(gpu => {
            const gpuCard = document.createElement('div');
            gpuCard.className = 'gpu-card mb-3';
            gpuCard.innerHTML = `
                <h6>GPU ${gpu.id}: ${gpu.name}</h6>
                <div class="progress mb-2">
                    <div class="progress-bar" role="progressbar" 
                         style="width: ${Number(gpu.utilization).toFixed(1)}%"
                         aria-valuenow="${gpu.utilization}" 
                         aria-valuemin="0" 
                         aria-valuemax="100">
                        ${Number(gpu.utilization).toFixed(1)}%
                    </div>
                </div>
                <div class="d-flex justify-content-between">
                    <span>온도: ${Number(gpu.temperature).toFixed(1)}°C</span>
                    <span>메모리: ${Number(gpu.memory_used).toFixed(1)}MB / ${Number(gpu.memory_total).toFixed(1)}MB</span>
                </div>
                <div class="d-flex justify-content-between">
                    <span>전력: ${Number(gpu.power_draw).toFixed(1)}W / ${Number(gpu.power_limit).toFixed(1)}W</span>
                </div>
            `;
            gpuStatusDiv.appendChild(gpuCard);
        });
    } catch (error) {
        console.error('GPU 상태 업데이트 실패:', error);
        const gpuStatusDiv = document.getElementById('gpuStatus');
        gpuStatusDiv.innerHTML = '<div class="alert alert-danger">GPU 상태 정보를 가져오는데 실패했습니다.</div>';
    }
}

// 냉각 시스템 상태 업데이트
async function updateCoolingStatus() {
    try {
        const response = await fetch(`${API_BASE_URL}/api/cooling/status`);
        if (!response.ok) {
            throw new Error(`HTTP error! status: ${response.status}`);
        }
        const coolingInfo = await response.json();
        
        const coolingStatusDiv = document.getElementById('coolingStatus');
        coolingStatusDiv.innerHTML = '';
        
        if (typeof coolingInfo !== 'object' || coolingInfo === null) {
            throw new Error('냉각 시스템 정보가 올바른 형식이 아닙니다.');
        }
        
        Object.entries(coolingInfo).forEach(([sensor, temp]) => {
            const tempCard = document.createElement('div');
            tempCard.className = 'temp-card mb-2';
            tempCard.innerHTML = `
                <div class="d-flex justify-content-between">
                    <span>${sensor}</span>
                    <span>${Number(temp).toFixed(1)}°C</span>
                </div>
                <div class="progress">
                    <div class="progress-bar" role="progressbar" 
                         style="width: ${(Number(temp) / 100) * 100}%"
                         aria-valuenow="${temp}" 
                         aria-valuemin="0" 
                         aria-valuemax="100">
                    </div>
                </div>
            `;
            coolingStatusDiv.appendChild(tempCard);
        });
    } catch (error) {
        console.error('냉각 시스템 상태 업데이트 실패:', error);
        const coolingStatusDiv = document.getElementById('coolingStatus');
        coolingStatusDiv.innerHTML = '<div class="alert alert-danger">냉각 시스템 상태 정보를 가져오는데 실패했습니다.</div>';
    }
}

// 온도 그래프 업데이트
async function updateTemperatureChart() {
    try {
        const response = await fetch(`${API_BASE_URL}/api/metrics/history?hours=24`);
        if (!response.ok) {
            throw new Error(`HTTP error! status: ${response.status}`);
        }
        const data = await response.json();
        
        if (!Array.isArray(data)) {
            throw new Error('온도 데이터가 배열 형태가 아닙니다.');
        }
        
        // 데이터 처리
        const timestamps = [];
        const gpuTemps = [];
        const coolingTemps = [];
        
        data.forEach(record => {
            if (record._measurement === 'gpu_metrics') {
                timestamps.push(new Date(record._time).toLocaleTimeString('ko-KR', { 
                    hour: '2-digit', 
                    minute: '2-digit',
                    hour12: false 
                }));
                gpuTemps.push(Number(record._value));
            } else if (record._measurement === 'cooling_metrics') {
                coolingTemps.push(Number(record._value));
            }
        });
        
        // 차트 업데이트
        if (!temperatureChart) {
            const ctx = document.getElementById('temperatureChart').getContext('2d');
            temperatureChart = new Chart(ctx, {
                type: 'line',
                data: {
                    labels: timestamps,
                    datasets: [
                        {
                            label: 'GPU 온도',
                            data: gpuTemps,
                            borderColor: 'rgb(255, 99, 132)',
                            tension: 0.1
                        },
                        {
                            label: '시스템 온도',
                            data: coolingTemps,
                            borderColor: 'rgb(54, 162, 235)',
                            tension: 0.1
                        }
                    ]
                },
                options: {
                    responsive: true,
                    scales: {
                        x: {
                            title: {
                                display: true,
                                text: '시간'
                            }
                        },
                        y: {
                            beginAtZero: true,
                            title: {
                                display: true,
                                text: '온도 (°C)'
                            }
                        }
                    }
                }
            });
        } else {
            temperatureChart.data.labels = timestamps;
            temperatureChart.data.datasets[0].data = gpuTemps;
            temperatureChart.data.datasets[1].data = coolingTemps;
            temperatureChart.update();
        }
    } catch (error) {
        console.error('온도 그래프 업데이트 실패:', error);
        const chartContainer = document.getElementById('temperatureChart').parentElement;
        chartContainer.innerHTML = '<div class="alert alert-danger">온도 데이터를 가져오는데 실패했습니다.</div>';
    }
}

// 알림 업데이트
async function updateAlerts() {
    try {
        const response = await fetch(`${API_BASE_URL}/api/alerts`);
        if (!response.ok) {
            throw new Error(`HTTP error! status: ${response.status}`);
        }
        const alerts = await response.json();
        
        const alertsDiv = document.getElementById('alerts');
        alertsDiv.innerHTML = '';
        
        if (!Array.isArray(alerts)) {
            throw new Error('알림 데이터가 배열 형태가 아닙니다.');
        }
        
        if (alerts.length === 0) {
            alertsDiv.innerHTML = '<p class="text-muted">현재 알림이 없습니다.</p>';
            return;
        }
        
        alerts.forEach(alert => {
            const alertCard = document.createElement('div');
            alertCard.className = `alert alert-${getAlertType(alert.type)} mb-2`;
            alertCard.innerHTML = alert.message;
            alertsDiv.appendChild(alertCard);
        });
    } catch (error) {
        console.error('알림 업데이트 실패:', error);
        const alertsDiv = document.getElementById('alerts');
        alertsDiv.innerHTML = '<div class="alert alert-danger">알림 정보를 가져오는데 실패했습니다.</div>';
    }
}

// 알림 타입에 따른 스타일 반환
function getAlertType(type) {
    const alertTypes = {
        'temperature': 'danger',
        'memory': 'warning',
        'power': 'warning',
        'system': 'info'
    };
    return alertTypes[type] || 'secondary';
} 