from fastapi import FastAPI, Request
from fastapi.templating import Jinja2Templates
from fastapi.staticfiles import StaticFiles
from fastapi.responses import JSONResponse
from loguru import logger
import os
import sys

# src 디렉토리를 Python 경로에 추가
sys.path.append(os.path.dirname(os.path.dirname(os.path.dirname(os.path.abspath(__file__)))))

from src.monitors.gpu_monitor import GPUMonitor
from src.monitors.cooling_monitor import CoolingMonitor
from src.utils.influxdb_client import InfluxDBManager
from src.notifications.alert_manager import AlertManager

# FastAPI 앱 초기화
app = FastAPI(title="GPU 모니터링 대시보드")

# 정적 파일 및 템플릿 설정
app.mount("/static", StaticFiles(directory="src/web/static"), name="static")
templates = Jinja2Templates(directory="src/web/templates")

# 모니터링 객체 초기화
gpu_monitor = GPUMonitor()
cooling_monitor = CoolingMonitor()
influxdb_manager = InfluxDBManager()
alert_manager = AlertManager()

@app.get("/")
async def read_root(request: Request):
    """메인 대시보드 페이지 렌더링"""
    try:
        return templates.TemplateResponse(
            "index.html",
            {"request": request, "title": "GPU 모니터링 대시보드"}
        )
    except Exception as e:
        logger.error(f"대시보드 렌더링 중 오류 발생: {str(e)}")
        return JSONResponse(
            status_code=500,
            content={"error": "대시보드 렌더링 중 오류가 발생했습니다."}
        )

@app.get("/api/gpu/status")
async def get_gpu_status():
    """현재 GPU 상태 반환"""
    try:
        status = gpu_monitor.get_status()
        return JSONResponse(content=status)
    except Exception as e:
        logger.error(f"GPU 상태 조회 중 오류 발생: {str(e)}")
        return JSONResponse(
            status_code=500,
            content={"error": "GPU 상태 조회 중 오류가 발생했습니다."}
        )

@app.get("/api/cooling/status")
async def get_cooling_status():
    """현재 냉각 시스템 상태 반환"""
    try:
        status = cooling_monitor.get_status()
        return JSONResponse(content=status)
    except Exception as e:
        logger.error(f"냉각 시스템 상태 조회 중 오류 발생: {str(e)}")
        return JSONResponse(
            status_code=500,
            content={"error": "냉각 시스템 상태 조회 중 오류가 발생했습니다."}
        )

@app.get("/api/metrics/history")
async def get_metrics_history(hours: int = 24):
    """지정된 시간 범위의 메트릭 히스토리 반환"""
    try:
        history = influxdb_manager.get_metrics_history(hours)
        return JSONResponse(content=history)
    except Exception as e:
        logger.error(f"메트릭 히스토리 조회 중 오류 발생: {str(e)}")
        return JSONResponse(
            status_code=500,
            content={"error": "메트릭 히스토리 조회 중 오류가 발생했습니다."}
        )

@app.get("/api/alerts")
async def get_alerts():
    """현재 알림 상태 반환"""
    try:
        alerts = alert_manager.get_active_alerts()
        return JSONResponse(content=alerts)
    except Exception as e:
        logger.error(f"알림 상태 조회 중 오류 발생: {str(e)}")
        return JSONResponse(
            status_code=500,
            content={"error": "알림 상태 조회 중 오류가 발생했습니다."}
        ) 