import schedule
import time
from loguru import logger
from datetime import datetime
from typing import List, Dict

from monitors.gpu_monitor import GPUMonitor
from monitors.cooling_monitor import CoolingMonitor
from notifications.slack_notifier import SlackNotifier
from config.config import settings

# 로깅 설정
logger.add(
    settings.LOG_FILE,
    rotation="500 MB",
    retention="10 days",
    level=settings.LOG_LEVEL
)

class GPUMonitoringSystem:
    def __init__(self):
        self.gpu_monitor = GPUMonitor()
        self.cooling_monitor = CoolingMonitor()
        self.slack_notifier = SlackNotifier()
        logger.info("GPU 모니터링 시스템 초기화 완료")

    def check_system_status(self):
        try:
            # GPU 상태 확인
            gpu_alerts = self.gpu_monitor.check_gpu_status()
            
            # 냉각 시스템 상태 확인
            cooling_alerts = self.cooling_monitor.check_cooling_status()
            
            # 모든 알림 수집
            all_alerts = []
            
            # GPU 알림에 타임스탬프 추가
            for alert in gpu_alerts:
                alert['timestamp'] = datetime.now().strftime("%Y-%m-%d %H:%M:%S")
                all_alerts.append(alert)
            
            # 냉각 시스템 알림에 타임스탬프 추가
            for alert in cooling_alerts:
                alert['timestamp'] = datetime.now().strftime("%Y-%m-%d %H:%M:%S")
                all_alerts.append(alert)
            
            # 알림 전송
            if all_alerts:
                self.slack_notifier.send_alert(all_alerts)
                logger.info(f"{len(all_alerts)}개의 알림 전송 완료")
            
        except Exception as e:
            logger.error(f"시스템 상태 확인 중 오류 발생: {str(e)}")

def main():
    try:
        monitoring_system = GPUMonitoringSystem()
        
        # 스케줄러 설정
        schedule.every(settings.MONITORING_INTERVAL).seconds.do(
            monitoring_system.check_system_status
        )
        
        logger.info("모니터링 시스템 시작")
        
        # 메인 루프
        while True:
            schedule.run_pending()
            time.sleep(1)
            
    except KeyboardInterrupt:
        logger.info("모니터링 시스템 종료")
    except Exception as e:
        logger.error(f"시스템 실행 중 오류 발생: {str(e)}")

if __name__ == "__main__":
    main() 