import logging
from flask import Flask, request, Response, stream_with_context
from flask_cors import CORS
import requests
import os
from werkzeug.middleware.proxy_fix import ProxyFix

app = Flask(__name__)
app.wsgi_app = ProxyFix(app.wsgi_app)  # ProxyFix 미들웨어 추가
CORS(app, resources={r"/*": {"origins": "*"}})  # 모든 출처에 대해 CORS를 활성화합니다.

# 로그 디렉터리 및 파일 설정
log_dir = '/app'
if not os.path.exists(log_dir):
    os.makedirs(log_dir)  # 로그 디렉터리가 없으면 생성
log_file = os.path.join(log_dir, 'proxy_app.log')

# 로깅 설정
logging.basicConfig(filename=log_file, level=logging.DEBUG,
                    format='%(asctime)s %(levelname)s %(name)s %(threadName)s : %(message)s')
app.logger.addHandler(logging.StreamHandler())
app.logger.setLevel(logging.DEBUG)

@app.before_request
def before_request():
    if request.scheme == 'http':
        # HTTP 요청을 HTTPS로 리다이렉트
        return Response(f"Redirecting to HTTPS: {request.url.replace('http://', 'https://')}", 
                        status=301, 
                        headers={'Location': request.url.replace('http://', 'https://')})

@app.route('/proxy', methods=['GET', 'POST', 'PUT', 'DELETE'])
def proxy_prusa_link():
    try:
        # 쿼리 파라미터에서 URL 가져오기
        url = request.args.get('url')
        if not url:
            return Response("URL is required", status=400)

        # 타겟 URL에 대한 요청 준비
        headers = {key: value for key, value in request.headers.items() if key != 'Host'}
        response = requests.request(
            method=request.method,
            url=url,
            headers=headers,
            data=request.get_data(),
            cookies=request.cookies,
            stream=True  # 스트리밍 응답 활성화
        )

        # 응답을 클라이언트로 스트리밍
        def generate():
            for chunk in response.iter_content(chunk_size=8192):
                yield chunk

        proxy_response = Response(stream_with_context(generate()), status=response.status_code)
        for key, value in response.headers.items():
            if key.lower() != 'content-encoding' and key.lower() != 'transfer-encoding':
                proxy_response.headers[key] = value

        return proxy_response
    except Exception as e:
        app.logger.error(f"An error occurred: {str(e)}")  # 에러 로그 기록
        return Response(f"An error occurred: {str(e)}", status=500)

if __name__ == '__main__':
    # SSL 인증서와 키 파일 경로 설정
    context = ('./proxy.geumdo.net/cert.pem', './proxy.geumdo.net/privkey.pem')
    app.run(host='0.0.0.0', port=3000, debug=True, ssl_context=context)  # SSL 컨텍스트 추가

